Currently libcpucycles supports the following cycle counters. Some
cycle counters are actually other forms of counters that libcpucycles
scales to imitate a cycle counter. There is
[separate documentation](selection.html)
for how libcpucycles makes a choice of cycle counter. See also
[security considerations](security.html) regarding enabling or disabling
counters and regarding Turbo Boost.

`amd64-pmc`: Requires a 64-bit Intel/AMD platform. Requires the Linux
perf_event interface. Accesses a cycle counter through RDPMC. Requires
`/proc/sys/kernel/perf_event_paranoid` to be at most 2 for user-level
RDPMC access. This counter runs at the clock frequency of the CPU core.

`amd64-tsc`, `amd64-tscasm`: Requires a 64-bit Intel/AMD platform.
Requires RDTSC to be enabled, which it is by default. Uses RDTSC to
access the CPU's time-stamp counter. On current CPUs, this is an
off-core clock rather than a cycle counter, but it is typically a very
fast off-core clock, making it adequate for seeing cycle counts if
overclocking and underclocking are disabled. The difference between
`tsc` and `tscasm` is that `tsc` uses the compiler's `__rdtsc()` while
`tscasm` uses inline assembly.

`arm32-cortex`: Requires a 32-bit ARMv7-A platform. Uses
`mrc p15, 0, %0, c9, c13, 0` to read the cycle counter. Requires user
access to the cycle counter, which is not enabled by default but can (sometimes) be
enabled under Linux via
[a kernel module](https://github.com/thoughtpolice/enable_arm_pmu).
This counter is natively 32 bits, but libcpucycles watches how the
counter and `gettimeofday` increase to compute a 64-bit extension of the
counter.

`arm32-1176`: Requires a 32-bit ARM1176 platform. Uses
`mrc p15, 0, %0, c15, c12, 1` to read the cycle counter. Requires user
access to the cycle counter, which is not enabled by default but can (sometimes) be
enabled under Linux via
[a kernel module](https://bench.cr.yp.to/cpucycles/n810.html).
This counter is natively 32 bits, but libcpucycles watches how the
counter and `gettimeofday` increase to compute a 64-bit extension of the
counter.

`arm64-pmc`: Requires a 64-bit ARMv8-A platform. Uses
`mrs %0, PMCCNTR_EL0` to read the cycle counter. Requires user access
to the cycle counter, which is not enabled by default but can be enabled
under Linux via
[a kernel module](https://github.com/rdolbeau/enable_arm_pmu).

`arm64-vct`: Requires a 64-bit ARMv8-A platform. Uses
`mrs %0, CNTVCT_EL0` to read a "virtual count" timer. This is an
off-core clock, typically running at 24MHz. Results are scaled by
libcpucycles.

`loong64-rdtime`: Requires a 64-bit LoongArch platform.
Uses `rdtime.d` to read the "constant frequency timer".
This is an off-core clock, typically running at 100MHz.

`mips64-cc`: Requires a 64-bit MIPS platform. (Maybe the same code would
also work as `mips32-cc`, but this has not been tested yet.) Uses RDHWR
to read the hardware cycle counter (hardware register 2 times a constant
scale factor in hardware register 3). This counter is natively 32 bits,
but libcpucycles watches how the counter and `gettimeofday` increase to
compute a 64-bit extension of the counter.

`ppc32-mftb`: Requires a 32-bit PowerPC platform. Uses `mftb` and
`mftbu` to read the "time base". This is an off-core clock, typically
running at 24MHz.

`ppc64-mftb`: Requires a 64-bit PowerPC platform. Uses `mftb` and
`mftbu` to read the "time base". This is an off-core clock, typically
running at 24MHz.

`riscv32-rdcycle`: Requires a 32-bit RISC-V platform. Uses `rdcycle`
and `rdcycleh` to read a cycle counter.

`riscv64-rdcycle`: Requires a 64-bit RISC-V platform. Uses `rdcycle`
to read a cycle counter.

`s390x-stckf`: Requires a 64-bit z/Architecture platform. Uses `stckf`
to read the TOD clock, which is documented to run at 4096MHz. On the
z15, this looks like a doubling of an off-core 2048MHz clock. Results
are scaled by libcpucycles.

`sparc64-rdtick`: Requires a 64-bit SPARC platform. Uses `rd %tick`
to read a cycle counter.

`x86-tsc`, `x86-tscasm`: Same as `amd64-tsc` and `amd64-tscasm`, but
for 32-bit Intel/AMD platforms instead of 64-bit Intel/AMD platforms.

`default-gettimeofday`: Reasonably portable. Resolution is limited to 1
microsecond. Results are scaled by libcpucycles.

`default-mach`: Requires an OS with `mach_absolute_time()`. Typically
runs at 24MHz. Results are scaled by libcpucycles.

`default-monotonic`: Requires `CLOCK_MONOTONIC`. Reasonably portable,
although might fail on older systems where `default-gettimeofday` works.
Resolution is limited to 1 nanosecond. Can be almost as good as a cycle
counter, or orders of magnitude worse, depending on the OS and CPU.
Results are scaled by libcpucycles.

`default-perfevent`: Requires the Linux `perf_event` interface, and a
CPU where `perf_event` supports `PERF_COUNT_HW_CPU_CYCLES`. Similar
variations in quality to `default-monotonic`, without the 1-nanosecond
limitation.

`default-zero`: The horrifying last resort if nothing else works.

## Examples

These are examples of `cpucycles-info` output on various machines. The
machines named `cfarm*` are from the
[GCC Compile Farm](https://gcc.gnu.org/wiki/CompileFarm).

A `median` line saying, e.g., `47 +47+28+0+2-5+0+2-5...` means that the
differences between adjacent cycle counts were 47+47, 47+28, 47+0, 47+2,
47−5, 47+0, 47+2, 47−5, etc., with median difference 47. The first few
differences are typically larger because of cache effects.
Current versions of libcpucycles use `iqm` (interquartile mean)
instead of `median`.

`berry0`,
Broadcom BCM2835:
```
cpucycles version 20240114
cpucycles tracesetup 0 arm32-cortex precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm32-1176 precision 22 scaling 1.000000 only32 1
cpucycles tracesetup 2 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 1199 scaling 1.000000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1200 scaling 1000.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1000000000
cpucycles implementation arm32-1176
cpucycles median 720 +942+124+1+0+2+0+0+0+0+0+0+0+0+0+0+0+0+0+0+1+2+0+0+2+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+222+300+1+0+0+2+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 798307692...2045181819 with 1024 loops 12 microseconds
cpucycles observed persecond 915478260...1260523810 with 2048 loops 22 microseconds
cpucycles observed persecond 947809523...1106100000 with 4096 loops 41 microseconds
cpucycles observed persecond 966353658...1129037500 with 8192 loops 81 microseconds
cpucycles observed persecond 988490566...1030019109 with 16384 loops 158 microseconds
cpucycles observed persecond 995169327...1002034063 with 32768 loops 2379 microseconds
cpucycles observed persecond 996871019...1012568691 with 65536 loops 627 microseconds
cpucycles observed persecond 997832134...1004212170 with 131072 loops 1250 microseconds
cpucycles observed persecond 997740918...1000887780 with 262144 loops 5009 microseconds
cpucycles observed persecond 998528349...1001961164 with 524288 loops 5537 microseconds
cpucycles observed persecond 999202882...1001166794 with 1048576 loops 10547 microseconds
```

`pi3aplus`,
Broadcom BCM2837B0:
```
cpucycles version 20250925
cpucycles tracesetup 0 arm64-pmc precision 9 scaling 1.000000 only32 0
cpucycles tracesetup 1 arm64-vct precision 173 scaling 73.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 191 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 272 scaling 1.400000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 1600 scaling 1400.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1400000000
cpucycles implementation arm64-pmc
cpucycles iqm 10 +10+8+3+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 1034250000...2162500000 with 2048 loops 3 microseconds
cpucycles observed persecond 1176285714...1696600000 with 4096 loops 6 microseconds
cpucycles observed persecond 1263461538...1513454546 with 8192 loops 12 microseconds
cpucycles observed persecond 1059225806...1141275863 with 16384 loops 30 microseconds
cpucycles observed persecond 1366166666...1432043479 with 32768 loops 47 microseconds
cpucycles observed persecond 1380052631...1412376345 with 65536 loops 94 microseconds
cpucycles observed persecond 1187592760...1199757991 with 131072 loops 220 microseconds
cpucycles observed persecond 1369133159...1377052494 with 262144 loops 382 microseconds
cpucycles observed persecond 1385285336...1389268875 with 524288 loops 756 microseconds
cpucycles observed persecond 1371701111...1373700066 with 1048576 loops 1528 microseconds
```

`bblack`,
TI Sitara XAM3359AZCZ100:
```
cpucycles version 20230105
cpucycles tracesetup 0 arm32-cortex precision 8 scaling 1.000000 only32 1
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 1283 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 1200 scaling 1000.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1000000000
cpucycles implementation arm32-cortex
cpucycles median 1260 +1506+62+31+7+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+13+7+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 622181818...2101888889 with 1024 loops 10 microseconds
cpucycles observed persecond 806133333...1492615385 with 2048 loops 14 microseconds
cpucycles observed persecond 879880000...1232565218 with 4096 loops 24 microseconds
cpucycles observed persecond 939577777...1130581396 with 8192 loops 44 microseconds
cpucycles observed persecond 956954022...1050047059 with 16384 loops 86 microseconds
cpucycles observed persecond 982878542...1020685715 with 32768 loops 246 microseconds
cpucycles observed persecond 988105105...1012217523 with 65536 loops 332 microseconds
cpucycles observed persecond 993752077...1007159723 with 131072 loops 721 microseconds
cpucycles observed persecond 995364296...1004009448 with 262144 loops 1377 microseconds
cpucycles observed persecond 998216306...1001821536 with 524288 loops 2685 microseconds
cpucycles observed persecond 998991848...1000914196 with 1048576 loops 5397 microseconds
```

`titan0`,
Intel Xeon E3-1275 V3:
```
cpucycles version 20250925
cpucycles tracesetup 0 amd64-pmc precision 41 scaling 1.000000 only32 0
cpucycles tracesetup 1 amd64-tsc precision 124 scaling 1.000000 only32 0
cpucycles tracesetup 2 amd64-tscasm precision 124 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 187 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 270 scaling 3.500000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 3700 scaling 3500.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3500000000
cpucycles implementation amd64-pmc
cpucycles iqm 45 +22+20+20+23-4+4-3-1+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 2064250000...4226000000 with 8192 loops 3 microseconds
cpucycles observed persecond 2755500000...4190000000 with 16384 loops 5 microseconds
cpucycles observed persecond 2991090909...3677444445 with 32768 loops 10 microseconds
cpucycles observed persecond 3283850000...3659666667 with 65536 loops 19 microseconds
cpucycles observed persecond 3452105263...3649444445 with 131072 loops 37 microseconds
cpucycles observed persecond 3407259740...3500720000 with 262144 loops 76 microseconds
cpucycles observed persecond 3472940397...3520859061 with 524288 loops 150 microseconds
cpucycles observed persecond 3472490066...3496286667 with 1048576 loops 301 microseconds
```

`nucnuc`,
Intel Pentium N3700:
```
cpucycles version 20250925
cpucycles tracesetup 0 amd64-pmc precision 26 scaling 1.000000 only32 0
cpucycles tracesetup 1 amd64-tsc precision 120 scaling 1.000000 only32 0
cpucycles tracesetup 2 amd64-tscasm precision 120 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 464 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 320 scaling 1.600000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 1800 scaling 1600.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1600000000
cpucycles implementation amd64-pmc
cpucycles iqm 66 +12+0+0+14+14+15+18-1+0+1-1+0-1+0-1+0-2+0-1+0-1+0-1+0-2+0-1+0-1+0-1+0-2+0-1+0-1+0-1+0-2+0-1+0-1+0-1+0-2+0-1+0-1+0+0-1+0+0-1+0-1+0-1+0
cpucycles observed persecond 1061250000...2323000000 with 2048 loops 3 microseconds
cpucycles observed persecond 1193571428...1779200000 with 4096 loops 6 microseconds
cpucycles observed persecond 1374666666...1705500000 with 8192 loops 11 microseconds
cpucycles observed persecond 1565714285...1761473685 with 16384 loops 20 microseconds
cpucycles observed persecond 1563952380...1656600000 with 32768 loops 41 microseconds
cpucycles observed persecond 1580530120...1626222223 with 65536 loops 82 microseconds
cpucycles observed persecond 1589430303...1612220859 with 131072 loops 164 microseconds
cpucycles observed persecond 1593920972...1605376147 with 262144 loops 328 microseconds
cpucycles observed persecond 1596176560...1601864123 with 524288 loops 656 microseconds
cpucycles observed persecond 1583584150...1588635677 with 1048576 loops 1324 microseconds
```

`saber214`,
AMD FX-8350:
```
cpucycles version 20250925
cpucycles tracesetup 0 amd64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 amd64-tsc precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 2 amd64-tscasm precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 368 scaling 4.013176 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 4213 scaling 4013.176000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 4013176000
cpucycles implementation amd64-tsc
cpucycles iqm 71 +45+0+0+0+0+1+0+0+0+0+1+0+0+0+1+0+0+0+0+1+0+0+0+0+1+0+0+0+1+0+0+0+0+1+0+0+0+1+0+0+0+0+1+0+0+0+1+0+0+0+0+1+0+0+0+1+0+0+0+0+1+0+0+0
cpucycles observed persecond 3017500000...6235500000 with 4096 loops 3 microseconds
cpucycles observed persecond 3442000000...4911200000 with 8192 loops 6 microseconds
cpucycles observed persecond 3697692307...4411909091 with 16384 loops 12 microseconds
cpucycles observed persecond 3840960000...4194956522 with 32768 loops 24 microseconds
cpucycles observed persecond 3916938775...4093425532 with 65536 loops 48 microseconds
cpucycles observed persecond 3956103092...4044252632 with 131072 loops 96 microseconds
cpucycles observed persecond 3996708333...4041210527 with 262144 loops 191 microseconds
cpucycles observed persecond 4003315000...4024698493 with 524288 loops 399 microseconds
cpucycles observed persecond 4006685378...4017770943 with 1048576 loops 765 microseconds
```

`cfarm14`,
Intel Xeon E5-2620 v3,
Debian 12,
Linux kernel 6.1.0-26-amd64:
```
cpucycles version 20250925
cpucycles tracesetup 0 amd64-pmc precision 41 scaling 1.000000 only32 0
cpucycles tracesetup 1 amd64-tsc precision 124 scaling 1.000000 only32 0
cpucycles tracesetup 2 amd64-tscasm precision 124 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-perfevent precision 160 scaling 1.000000 only32 0
cpucycles tracesetup 4 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 5 default-monotonic precision 289 scaling 3.200000 only32 0
cpucycles tracesetup 6 default-gettimeofday precision 3400 scaling 3200.000000 only32 0
cpucycles tracesetup 7 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3200000000
cpucycles implementation amd64-pmc
cpucycles iqm 46 +17+19+19+19-5-4-3+1+3-4+1+2-3+1+2-3+1+2-3+1+4+2-3+1+2-3+1+2-5-3-2+17-5-3+2-3+1+2-3+1+2-3+1+2-3+1+2-3+1+2-3+1+2-3+1+2-3+1+2-3+1+2-3+1
cpucycles observed persecond 1654400000...2820333334 with 8192 loops 4 microseconds
cpucycles observed persecond 1835888888...2391857143 with 16384 loops 8 microseconds
cpucycles observed persecond 2057125000...2364928572 with 32768 loops 15 microseconds
cpucycles observed persecond 2053531250...2196733334 with 65536 loops 31 microseconds
cpucycles observed persecond 2050453125...2119612904 with 131072 loops 63 microseconds
cpucycles observed persecond 2081849206...2116951613 with 262144 loops 125 microseconds
cpucycles observed persecond 2089470119...2107012049 with 524288 loops 250 microseconds
cpucycles observed persecond 2093303393...2102072145 with 1048576 loops 500 microseconds
```

`cfarm26`,
Intel Core i5-4570 in 32-bit mode under KVM,
Debian 12.12,
Linux kernel 6.1.0-39-686-pae:
```
cpucycles version 20250925
cpucycles tracesetup 0 x86-tsc precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 1 x86-tscasm precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 590 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 2476 scaling 3.192606 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3392 scaling 3192.606000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3192606000
cpucycles implementation x86-tsc
cpucycles iqm 21 +19+6+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3
cpucycles observed persecond 1947500000...6185000000 with 8192 loops 3 microseconds
cpucycles observed persecond 2585500000...5025500000 with 16384 loops 5 microseconds
cpucycles observed persecond 2587666666...3564500000 with 32768 loops 11 microseconds
cpucycles observed persecond 3036820000...3292583334 with 65536 loops 49 microseconds
cpucycles observed persecond 3116800000...3280051283 with 131072 loops 79 microseconds
cpucycles observed persecond 3132593103...3223419581 with 262144 loops 144 microseconds
cpucycles observed persecond 3164520295...3219226766 with 524288 loops 270 microseconds
cpucycles observed persecond 3182085551...3209253817 with 1048576 loops 525 microseconds
```

`cfarm27`,
Intel Core i5-4570 in 32-bit mode under KVM,
Alpine 3.19.8,
Linux kernel 6.6.103-0-lts:
```
cpucycles version 20250925
cpucycles tracesetup 0 x86-tsc precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 1 x86-tscasm precision 118 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 634 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 1109 scaling 3.192606 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3392 scaling 3192.606000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3192606000
cpucycles implementation x86-tsc
cpucycles iqm 21 +13+259-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3-3+10-3+10-3-3+10-3-3+10-3
cpucycles observed persecond 1933000000...4836500000 with 4096 loops 3 microseconds
cpucycles observed persecond 2574166666...4361250000 with 8192 loops 5 microseconds
cpucycles observed persecond 2806000000...3672111112 with 16384 loops 10 microseconds
cpucycles observed persecond 2939095238...3361052632 with 32768 loops 20 microseconds
cpucycles observed persecond 3067228571...3318030304 with 65536 loops 34 microseconds
cpucycles observed persecond 3129857142...3268098361 with 131072 loops 62 microseconds
cpucycles observed persecond 3163294871...3218266234 with 262144 loops 155 microseconds
cpucycles observed persecond 3172141176...3222511905 with 524288 loops 169 microseconds
cpucycles observed persecond 3183567741...3197336570 with 1048576 loops 619 microseconds
```

`cfarm29`,
IBM POWER9,
Debian 13.1,
Linux kernel 6.12.48+deb13-powerpc64le:
```
cpucycles version 20250925
cpucycles tracesetup 0 ppc64-mftb precision 220 scaling 7.500000 only32 0
cpucycles tracesetup 1 default-perfevent precision 378 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 375 scaling 3.800000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 4000 scaling 3800.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3800000000
cpucycles implementation ppc64-mftb
cpucycles iqm 229 +26+3-4-4+4-4+11+3-4+4+11-4+11+3-4-11+18-4+4-4-4+11+3-4+19+3-4+11-4-11+18-11-4+3-4+11+4-4+11-4-4+3-4+4+3-4-11+11+3+4+3-4+11-11-12+11-4-4+19+3-11+11+3-4
cpucycles observed persecond 3355400000...6075000000 with 2048 loops 4 microseconds
cpucycles observed persecond 3245625000...4581333334 with 4096 loops 7 microseconds
cpucycles observed persecond 3378214285...4066833334 with 8192 loops 13 microseconds
cpucycles observed persecond 3585000000...3885821429 with 16384 loops 29 microseconds
cpucycles observed persecond 3746333333...3952325582 with 32768 loops 44 microseconds
cpucycles observed persecond 3761370967...3900866667 with 65536 loops 61 microseconds
cpucycles observed persecond 3820262295...3890375000 with 131072 loops 121 microseconds
cpucycles observed persecond 3819069672...3853665290 with 262144 loops 243 microseconds
cpucycles observed persecond 3826154004...3843447423 with 524288 loops 486 microseconds
cpucycles observed persecond 3836012295...3844750514 with 1048576 loops 975 microseconds
```

`cfarm45`,
AMD Athlon II X4 640,
Debian 8.11,
Linux kernel 3.16.0-11-686-pae:
```
cpucycles version 20230105
cpucycles tracesetup 0 x86-tsc precision 199 scaling 1.000000 only32 0
cpucycles tracesetup 1 x86-tscasm precision 199 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 941 scaling 3.000000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3200 scaling 3000.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3000000000
cpucycles implementation default-perfevent
cpucycles median 72 +12+0+0+0+0+0+0+0+5+0+0+0+0+0+0+0+2+0+0+0+0+0+0+0+1+0+0+0+0+0+0+0+2+0+0+0+0+0+0+0+1+0+0+0+0+0+0+0+2+0+0+0+0+0+0+0+1+0+0+0+0+0+0
cpucycles observed persecond 541500000...1812000000 with 1024 loops 3 microseconds
cpucycles observed persecond 712333333...1212250000 with 2048 loops 5 microseconds
cpucycles observed persecond 1193285714...1733600000 with 4096 loops 6 microseconds
cpucycles observed persecond 1689176470...1804562500 with 8192 loops 33 microseconds
cpucycles observed persecond 1713074626...1770600000 with 16384 loops 66 microseconds
cpucycles observed persecond 1765107692...1795140625 with 32768 loops 129 microseconds
cpucycles observed persecond 1785369649...1800603922 with 65536 loops 256 microseconds
cpucycles observed persecond 1781377862...1796288462 with 131072 loops 261 microseconds
cpucycles observed persecond 1772647398...1778247827 with 262144 loops 691 microseconds
cpucycles observed persecond 1789670493...1794149598 with 524288 loops 870 microseconds
cpucycles observed persecond 1860276211...1861561332 with 1048576 loops 3156 microseconds
```

`cfarm91`,
StarFive JH7100,
Debian trixie/sid,
Linux kernel 5.18.11-starfive:
```
cpucycles version 20250925
cpucycles tracesetup 0 riscv64-rdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 2702 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 1351 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation default-monotonic
cpucycles iqm 1476 +828-324+60+60+60+60-324+60-324+60+60-324+60+60-324+60+60-324+60-324+60+60-324+62-324+60+60-324+60+60-324+60-324+60+60-324+60+60-324+60-324+60+60-324+60+60-324+60-324+60+60+60+60+60-324+60-324+62+60+444+60-324+60+60
cpucycles observed persecond 1440250000...3968333334 with 1024 loops 7 microseconds
cpucycles observed persecond 1920181818...3072222223 with 2048 loops 10 microseconds
cpucycles observed persecond 2162473684...2733294118 with 4096 loops 18 microseconds
cpucycles observed persecond 2218777777...2530000000 with 8192 loops 35 microseconds
cpucycles observed persecond 2338014705...2490318182 with 16384 loops 67 microseconds
cpucycles observed persecond 2358559701...2440840910 with 32768 loops 133 microseconds
cpucycles observed persecond 2380905660...2419452472 with 65536 loops 264 microseconds
cpucycles observed persecond 2390815939...2410163810 with 131072 loops 526 microseconds
cpucycles observed persecond 2393901140...2403580953 with 262144 loops 1051 microseconds
cpucycles observed persecond 2397546190...2402577217 with 524288 loops 2099 microseconds
cpucycles observed persecond 2398864140...2401570967 with 1048576 loops 4327 microseconds
```

`cfarm92`,
SiFive Freedom U740,
Ubuntu 22.04.3,
Linux kernel 5.19.0-1021-generic:
```
cpucycles version 20240114
cpucycles tracesetup 0 riscv64-rdcycle precision 8 scaling 1.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 2599 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation riscv64-rdcycle
cpucycles median 8 +168+20+2+2+0+0+0+0+570+0+0+0+0+0+0+0+144+0+0+0+0+0+0+0+160+0+0+0+0+0+0+0+160+0+0+0+0+0+0+0+154+0+0+0+0+0+0+0+154+0+0+0+0+0+0+0+152+0+0+0+0+0+0
cpucycles observed persecond 571500000...2198000000 with 1024 loops 3 microseconds
cpucycles observed persecond 833600000...2094000000 with 2048 loops 4 microseconds
cpucycles observed persecond 921888888...1445142858 with 4096 loops 8 microseconds
cpucycles observed persecond 1029625000...1320642858 with 8192 loops 15 microseconds
cpucycles observed persecond 1137034482...1284481482 with 16384 loops 28 microseconds
cpucycles observed persecond 1155701754...1227454546 with 32768 loops 56 microseconds
cpucycles observed persecond 1177464285...1217163637 with 65536 loops 111 microseconds
cpucycles observed persecond 1188018099...1207858448 with 131072 loops 220 microseconds
cpucycles observed persecond 1189925170...1200519363 with 262144 loops 440 microseconds
cpucycles observed persecond 1193962457...1199117446 with 524288 loops 878 microseconds
cpucycles observed persecond 1194051324...1196780111 with 1048576 loops 1811 microseconds
```

`cfarm95`,
SpacemiT X60,
Debian trixie/sid,
Linux kernel 6.6.36-cfarm #1:
```
cpucycles version 20250925
cpucycles tracesetup 0 riscv64-rdcycle precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 1855 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 265 scaling 1.600000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 1800 scaling 1600.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 1600000000
cpucycles implementation default-monotonic
cpucycles iqm 134 -1-2+1+331+1-1-67-1-2+1-1+398+1-1-2+1-1-2+1+331+1-1-2+1-1-2+1+199-2+1-1-2+1-1-2+266+1-1-2+1-1-2+1+199-2+1-1-2+1-1-2+1-1-2+1-1-2-66-2+1-1-2+1-1
cpucycles observed persecond 1389000000...2217000000 with 4096 loops 5 microseconds
cpucycles observed persecond 1389000000...1740000000 with 8192 loops 11 microseconds
cpucycles observed persecond 1568285714...1764894737 with 16384 loops 20 microseconds
cpucycles observed persecond 1573047619...1665050000 with 32768 loops 41 microseconds
cpucycles observed persecond 1565904761...1610585366 with 65536 loops 83 microseconds
cpucycles observed persecond 1581548192...1604097561 with 131072 loops 165 microseconds
cpucycles observed persecond 1594966565...1606550459 with 262144 loops 328 microseconds
cpucycles observed persecond 1599219512...1604923548 with 524288 loops 655 microseconds
cpucycles observed persecond 1598772187...1601569630 with 1048576 loops 1430 microseconds
```

`cfarm103`,
Apple M1 (Icestorm-M1 + Firestorm-M1),
Debian trixie/sid,
Linux kernel 6.5.0-asahi-00780-g62806c2c6f29:
```
cpucycles version 20250925
cpucycles tracesetup 0 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-vct precision 186 scaling 86.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 285 scaling 2.064000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 2264 scaling 2064.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2064000000
cpucycles implementation arm64-vct
cpucycles iqm 0 +86+0+0+0+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+86+0+0+0+0+0+0+0+0
cpucycles observed persecond 1655500000...3354000000 with 8192 loops 3 microseconds
cpucycles observed persecond 1867428571...2666000000 with 16384 loops 6 microseconds
cpucycles observed persecond 1873571428...2200166667 with 32768 loops 13 microseconds
cpucycles observed persecond 2007769230...2185833334 with 65536 loops 25 microseconds
cpucycles observed persecond 2006115384...2089800000 with 131072 loops 51 microseconds
cpucycles observed persecond 2027262135...2069960397 with 262144 loops 102 microseconds
cpucycles observed persecond 2056798029...2078119403 with 524288 loops 202 microseconds
cpucycles observed persecond 2061664197...2072322581 with 1048576 loops 404 microseconds
```

`cfarm104`,
Apple M1 (Icestorm-M1 + Firestorm-M1),
MacOSX 12.6 21.6.0:
```
cpucycles version 20250925
cpucycles tracesetup 0 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-vct precision 200 scaling 100.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-mach precision 200 scaling 100.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 2599 scaling 2.399988 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation arm64-vct
cpucycles iqm 0 +5200+0+0+0+0+0+0+0+0+0+0+100+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+100+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+100+0+0
cpucycles observed persecond 1750000000...3500000000 with 8192 loops 3 microseconds
cpucycles observed persecond 1885714285...2640000000 with 16384 loops 6 microseconds
cpucycles observed persecond 2225000000...2670000000 with 32768 loops 11 microseconds
cpucycles observed persecond 2308695652...2528571429 with 65536 loops 22 microseconds
cpucycles observed persecond 2337777777...2446511628 with 131072 loops 44 microseconds
cpucycles observed persecond 2365168539...2421839081 with 262144 loops 88 microseconds
cpucycles observed persecond 2379661016...2406857143 with 524288 loops 176 microseconds
cpucycles observed persecond 2392613636...2406285715 with 1048576 loops 351 microseconds
```

`cfarm110` (`gcc1-power7`),
IBM POWER7,
CentOS 7.9 AltArch,
Linux kernel 3.10.0-1160.105.1.el7.ppc64:
```
cpucycles version 20250925
cpucycles tracesetup 0 ppc64-mftb precision 212 scaling 7.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 223 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 310 scaling 3.550000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3750 scaling 3550.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3550000000
cpucycles implementation ppc64-mftb
cpucycles iqm 167 -6+36+29-6+8-20+22-20+22-48+36-34+29-27+29-34+15-13+15-13+8-6+1+1-6+8-13+15+29-27+15-13+1+1-20+22-34+36-48+36-41+29-27+8-6+29-27+43-41+22-20+8-6-13+15-27+29-41+29-41+29-27+8-6
cpucycles observed persecond 2450000000...5295500000 with 2048 loops 3 microseconds
cpucycles observed persecond 3035666666...4779250000 with 4096 loops 5 microseconds
cpucycles observed persecond 3249750000...3963400000 with 8192 loops 11 microseconds
cpucycles observed persecond 3359333333...3745736843 with 16384 loops 20 microseconds
cpucycles observed persecond 3416000000...3604125000 with 32768 loops 41 microseconds
cpucycles observed persecond 3468915254...3601438597 with 65536 loops 58 microseconds
cpucycles observed persecond 3540125000...3608881819 with 131072 loops 111 microseconds
cpucycles observed persecond 3571355855...3606018182 with 262144 loops 221 microseconds
cpucycles observed persecond 3579069977...3596444445 with 524288 loops 442 microseconds
cpucycles observed persecond 3578927765...3587547512 with 1048576 loops 885 microseconds
```

`cfarm112` (`gcc2-power8`),
IBM POWER8E,
CentOS 7.9 AltArch,
Linux kernel 3.10.0-1127.13.1.el7.ppc64le:
```
cpucycles version 20250925
cpucycles tracesetup 0 ppc64-mftb precision 267 scaling 7.250000 only32 0
cpucycles tracesetup 1 default-perfevent precision 333 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 565 scaling 3.690000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3890 scaling 3690.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3690000000
cpucycles implementation ppc64-mftb
cpucycles iqm 214 +4-4-4+4-18+3+4-4+3+105+11+4+3+4-4+3-11-3-4-4+4-11-4-4+4+3+69+11+3-4+4-4+4+3+4-11-4-11-4-4+11-4-11-3-4-4+4-19-3+3+4+3+4-4-11+11+3-4+4-11-18-4+3+11
cpucycles observed persecond 2440400000...6890000000 with 1024 loops 4 microseconds
cpucycles observed persecond 3287285714...4796600000 with 2048 loops 6 microseconds
cpucycles observed persecond 3446166666...4230400000 with 4096 loops 11 microseconds
cpucycles observed persecond 3505520000...3846913044 with 8192 loops 24 microseconds
cpucycles observed persecond 3541447368...3797000000 with 16384 loops 37 microseconds
cpucycles observed persecond 3635138461...3761476191 with 32768 loops 64 microseconds
cpucycles observed persecond 3689914062...3753833334 with 65536 loops 127 microseconds
cpucycles observed persecond 3688777343...3720535434 with 131072 loops 255 microseconds
cpucycles observed persecond 3702745098...3718651575 with 262144 loops 509 microseconds
cpucycles observed persecond 3709734774...3717701772 with 524288 loops 1017 microseconds
cpucycles observed persecond 3709592829...3713586038 with 1048576 loops 2035 microseconds
```

`cfarm120`,
IBM POWER10,
AlmaLinux 9.6,
Linux kernel 5.14.0-427.31.1.el9_4.ppc64le:
```
cpucycles version 20250925
cpucycles tracesetup 0 ppc64-mftb precision 123 scaling 5.750000 only32 0
cpucycles tracesetup 1 default-perfevent precision 205 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 226 scaling 2.950000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3150 scaling 2950.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2950000000
cpucycles implementation ppc64-mftb
cpucycles iqm 60 +15+20-31+9+15-32+9-31+15-37+20+15-37+15-31+14+9-31+21+9-32+15-37+15+15-37+20-31+9+15-32+15-37+15+15-37+20-31+9-31+14+9-31+15+15-32+15-37+15-31+9+14-31+15-31+14+9-31+15+15-32+9-31+15
cpucycles observed persecond 2454000000...4170666667 with 4096 loops 4 microseconds
cpucycles observed persecond 2609750000...3543000000 with 8192 loops 7 microseconds
cpucycles observed persecond 2827466666...3281461539 with 16384 loops 14 microseconds
cpucycles observed persecond 2840500000...3039066667 with 32768 loops 31 microseconds
cpucycles observed persecond 2899142857...3080424243 with 65536 loops 34 microseconds
cpucycles observed persecond 2905957142...2994808824 with 131072 loops 69 microseconds
cpucycles observed persecond 2905228571...2948710145 with 262144 loops 139 microseconds
cpucycles observed persecond 2940666666...2962824818 with 524288 loops 275 microseconds
cpucycles observed persecond 2935048648...2946007234 with 1048576 loops 554 microseconds
```

`cfarm185`,
Ampere eMAG 8180,
AlmaLinux 8.10,
Linux kernel 4.18.0-553.el8_10.aarch64:
```
cpucycles version 20250925
cpucycles tracesetup 0 arm64-pmc precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 1 arm64-vct precision 175 scaling 75.000000 only32 0
cpucycles tracesetup 2 default-perfevent precision 343 scaling 1.000000 only32 0
cpucycles tracesetup 3 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 4 default-monotonic precision 275 scaling 3.000000 only32 0
cpucycles tracesetup 5 default-gettimeofday precision 3200 scaling 3000.000000 only32 0
cpucycles tracesetup 6 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3000000000
cpucycles implementation arm64-vct
cpucycles iqm 40 +35+35-40-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35+35-40-40+35+35-40+35-40+35-40+35-40+35-40+35-40+35-40+35
cpucycles observed persecond 2812500000...5737500000 with 4096 loops 3 microseconds
cpucycles observed persecond 2803125000...3787500000 with 8192 loops 7 microseconds
cpucycles observed persecond 2798437500...3219642858 with 16384 loops 15 microseconds
cpucycles observed persecond 2886290322...3100862069 with 32768 loops 30 microseconds
cpucycles observed persecond 2932377049...3039406780 with 65536 loops 60 microseconds
cpucycles observed persecond 2955371900...3007563026 with 131072 loops 120 microseconds
cpucycles observed persecond 2979375000...3005357143 with 262144 loops 239 microseconds
cpucycles observed persecond 2997798742...3011052632 with 524288 loops 476 microseconds
cpucycles observed persecond 2997798742...3004332984 with 1048576 loops 953 microseconds
```

`cfarm202`,
UltraSparc T5,
Debian unstable (bookworm),
Linux kernel 5.19.0-2-sparc64-smp:
```
cpucycles version 20230105
cpucycles tracesetup 0 sparc64-rdtick precision 65 scaling 1.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 386 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 442 scaling 3.599910 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 3799 scaling 3599.910000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 3599910000
cpucycles implementation sparc64-rdtick
cpucycles median 73 +24+0+24+24+24+24+24+24+0+1+24+0+1+24+0+1+24+0+0+1+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+1+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 2751500000...4258250000 with 4096 loops 5 microseconds
cpucycles observed persecond 3289200000...4206875000 with 8192 loops 9 microseconds
cpucycles observed persecond 3454789473...3900823530 with 16384 loops 18 microseconds
cpucycles observed persecond 3452026315...3659888889 with 32768 loops 37 microseconds
cpucycles observed persecond 3543770270...3650916667 with 65536 loops 73 microseconds
cpucycles observed persecond 3567299319...3620662069 with 131072 loops 146 microseconds
cpucycles observed persecond 3591373287...3618220690 with 262144 loops 291 microseconds
cpucycles observed persecond 3597353344...3610774527 with 524288 loops 582 microseconds
cpucycles observed persecond 3595899403...3603058071 with 1048576 loops 1172 microseconds
```

`cfarm216`,
VM on SPARC-M8,
Solaris 11.4.85.201.2
(`tracesetup` not working):
```
cpucycles version 20250925
cpucycles persecond 5067000000
cpucycles implementation sparc64-rdtick
cpucycles iqm 69 +25+25+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+33330+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 4115600000...6986000000 with 8192 loops 4 microseconds
cpucycles observed persecond 4107300000...5188875000 with 16384 loops 9 microseconds
cpucycles observed persecond 4826529411...5495400000 with 32768 loops 16 microseconds
cpucycles observed persecond 4821735294...5135093750 with 65536 loops 33 microseconds
cpucycles observed persecond 4892208955...5048584616 with 131072 loops 66 microseconds
cpucycles observed persecond 4965590909...5044907693 with 262144 loops 131 microseconds
cpucycles observed persecond 4984102661...5023766284 with 524288 loops 262 microseconds
cpucycles observed persecond 5002935114...5022831418 with 1048576 loops 523 microseconds
```

`cfarm230`,
Cavium Octeon III V0.2,
Debian 10.13,
Linux kernel 4.9.79-UBNT_E300:
```
cpucycles version 20250925
cpucycles tracesetup 0 mips64-cc precision 23 scaling 1.000000 only32 1
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 1433 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation mips64-cc
cpucycles iqm 405 +34+37+12+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 776100000...1741125000 with 1024 loops 9 microseconds
cpucycles observed persecond 923000000...1198214286 with 2048 loops 15 microseconds
cpucycles observed persecond 938645161...1052379311 with 4096 loops 30 microseconds
cpucycles observed persecond 979084745...1038035088 with 8192 loops 58 microseconds
cpucycles observed persecond 983846153...1013147827 with 16384 loops 116 microseconds
cpucycles observed persecond 990508620...1005217392 with 32768 loops 231 microseconds
cpucycles observed persecond 996039045...1003522876 with 65536 loops 460 microseconds
cpucycles observed persecond 998831338...1002538714 with 131072 loops 918 microseconds
cpucycles observed persecond 998601741...1001037582 with 262144 loops 1837 microseconds
cpucycles observed persecond 999574618...1000599728 with 524288 loops 3671 microseconds
cpucycles observed persecond 999700338...1000298251 with 1048576 loops 7374 microseconds
```

`cfarm400`,
Loongson-3C5000L-LL,
Debian trixie/sid,
Linux kernel 6.1.0-rc7+:
```
cpucycles version 20250925
cpucycles tracesetup 0 loong64-rdtime precision 124 scaling 24.000000 only32 0
cpucycles tracesetup 1 default-perfevent precision 170 scaling 1.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 296 scaling 2.399988 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 2599 scaling 2399.987654 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 2399987654
cpucycles implementation loong64-rdtime
cpucycles iqm 48 +24+0+0+0+0+0+0+0+0+0+0+24+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+24+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+0+24+0+0+0+0+0+0+0+0+0+0+0+0
cpucycles observed persecond 1652000000...2556000000 with 4096 loops 5 microseconds
cpucycles observed persecond 2194666666...2869714286 with 8192 loops 8 microseconds
cpucycles observed persecond 2189333333...2482500000 with 16384 loops 17 microseconds
cpucycles observed persecond 2385454545...2549419355 with 32768 loops 32 microseconds
cpucycles observed persecond 2348776119...2425846154 with 65536 loops 66 microseconds
cpucycles observed persecond 2383636363...2422707693 with 131072 loops 131 microseconds
cpucycles observed persecond 2383454545...2402839695 with 262144 loops 263 microseconds
cpucycles observed persecond 2396891428...2406653920 with 524288 loops 524 microseconds
cpucycles observed persecond 2399107721...2403965617 with 1048576 loops 1048 microseconds
```

IBM z15:
```
cpucycles version 20230106
cpucycles tracesetup 0 s390x-stckf precision 250 scaling 1.269531 only32 0
cpucycles tracesetup 1 default-perfevent precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 2 default-mach precision 0 scaling 0.000000 only32 0
cpucycles tracesetup 3 default-monotonic precision 272 scaling 5.200000 only32 0
cpucycles tracesetup 4 default-gettimeofday precision 5400 scaling 5200.000000 only32 0
cpucycles tracesetup 5 default-zero precision 0 scaling 0.000000 only32 0
cpucycles persecond 5200000000
cpucycles implementation s390x-stckf
cpucycles median 48 +87+8+0-2+0+0+38-2+0+1-3+1+28+0+3-3+1+0+28+0-2+3+0-2+36+0+0+0+1+0+28+0-2+0+3-2+35+1+0-2+0+3+28+0-2+0+0-2+3+25+3+0-2+0+1+35+1+0+0-2+0+28+0
cpucycles observed persecond 4948941176...5627733334 with 8192 loops 16 microseconds
cpucycles observed persecond 4104125000...5515666667 with 16384 loops 7 microseconds
cpucycles observed persecond 5047076923...5987818182 with 32768 loops 12 microseconds
cpucycles observed persecond 5044846153...5475708334 with 65536 loops 25 microseconds
cpucycles observed persecond 5141313725...5357428572 with 131072 loops 50 microseconds
cpucycles observed persecond 5150892156...5257250000 with 262144 loops 101 microseconds
cpucycles observed persecond 5183421568...5236549505 with 524288 loops 203 microseconds
cpucycles observed persecond 5190282555...5216582717 with 1048576 loops 406 microseconds
```
