/**
 * \file pappsomspp/processing/specpeptidoms/scenario.h
 * \date 24/03/2025
 * \author Aurélien Berthier
 * \brief backtracking of 2nd alignment
 *
 * C++ implementation of the SpecPeptidOMS algorithm described in :
 * (1) Benoist, É.; Jean, G.; Rogniaux, H.; Fertin, G.; Tessier, D. SpecPeptidOMS Directly and
 * Rapidly Aligns Mass Spectra on Whole Proteomes and Identifies Peptides That Are Not Necessarily
 * Tryptic: Implications for Peptidomics. J. Proteome Res. 2025.
 * https://doi.org/10.1021/acs.jproteome.4c00870.
 */

/*
 * Copyright (c) 2025 Aurélien Berthier
 * <aurelien.berthier@ls2n.fr>
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <cstddef>
#include <vector>
#include "types.h"

namespace pappso
{
namespace specpeptidoms
{
struct ScenarioCell
{
  std::size_t previous_row, previous_column;
  AlignType alignment_type;
};

// struct InterpretationCell
// {
//   uint8_t aa;
//   double shift;
//   bool found_aa;
// };

class Scenario
{
  public:
  /**
   * Constructor
   */
  Scenario();

  /**
   * Destructor
   */
  ~Scenario();

  /**
   * @brief Stores the origin (cell location and alignment type) of the provided cell in the
   * backtracking matrix
   */
  void saveOrigin(std::size_t current_row,
                  std::size_t current_column,
                  std::size_t previous_row,
                  std::size_t previous_column,
                  int score,
                  AlignType alignment_type);

  /**
   * @brief Allocate new storage to the backtracking matrix if needed
   */
  void reserve(std::size_t n_rows, std::size_t n_columns);

  /**
   * @brief Returns the scenario cells corresponding to the best alignment and the best alignment's
   * score
   */
  std::pair<std::vector<ScenarioCell>, int> getBestAlignment() const;

  int getBestScore() const;

  void resetScenario();

  private:
  std::vector<std::vector<ScenarioCell>> m_origin_matrix;
  std::pair<ScenarioCell, int> m_best_alignment = {{0, 0, AlignType::init}, -15};
};
} // namespace specpeptidoms
} // namespace pappso
