/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/CrossLinkedRegion.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


//            [3] [4] [5] [6]        [9]    [11]
// o---o---o---o---o--o---o---o---o---o---o---o---o---o---o
//             |   |__|   |       |   |       |       |
//             |          +-----------+       +-------+
//             |                  |
//             +------------------+
//
//
// In the example above, there are two cross-linked regions: [3--9]
// and [11--13].


/*!
\class MsXpS::libXpertMassCore::CrossLinkedRegion
\inmodule libXpertMass
\ingroup PolChemDefAqueousChemicalReactions
\inheaderfile CrossLinkedRegion.hpp

\brief The CrossLinkedRegion class provides abstractions to work with
cross-links and detect the region of oligomers that are linked together.

In the following representation, a polymer sequence has a number of CrossLink
instances that are intertwined:

\code

           [3] [4] [5] [6]        [9]    [11]     [13]
o---o---o---o---o--o---o---o---o---o---o---o---o---o---o
            |   |__|   |       |   |       |       |
            |          +-----------+       +-------+
            |                  |
            +------------------+

\endcode

In this example, there are two cross-linked regions: [3--9]
and [11--13].

The first cross-linked region has a start index of 3 and a stop index of 9 and
involves three cross-links.

The second cross-linked region only involves a single cross-link involving two
monomers at indices 11 and 13.

This class is most useful when performing fragmentations of Oligomer instances.
*/

/*!
\variable MsXpS::libXpertMassCore::CrossLinkedRegion::m_startIndex

\brief The start index of this cross-linked region.
*/

/*!
\variable MsXpS::libXpertMassCore::CrossLinkedRegion::m_stopIndex

\brief The stop index of this cross-linked region.
*/

/*!
\variable MsXpS::libXpertMassCore::CrossLinkedRegion::m_crossLinks

\brief The CrossLink instances that are contained in this CrossLinkedRegion
instance.
*/


/*!
\brief
*/
CrossLinkedRegion::CrossLinkedRegion()
{
  m_startIndex = 0;
  m_stopIndex  = 0;
}

/*!
\brief Constructs a CrossLinkedRegion instance with indices \a index_start and
\a index_stop.
*/
CrossLinkedRegion::CrossLinkedRegion(std::size_t index_start,
                                     std::size_t index_stop)
{
  m_startIndex = index_start;
  m_stopIndex  = index_stop;
}

/*!
\brief Constructs a CrossLinkedRegion instance as a copy of \a other.

The copy of the CrossLink instances is shallow (the shared pointers are copied).
*/
CrossLinkedRegion::CrossLinkedRegion(const CrossLinkedRegion &other)
  : m_startIndex(other.m_startIndex),
    m_stopIndex(other.m_stopIndex),
    m_crossLinks(other.m_crossLinks)
{
}

/*!
\brief Destructs a CrossLinkedRegion instance.
*/
CrossLinkedRegion::~CrossLinkedRegion()
{
}

/*!
\brief Sets the start index of this instance to \a index.
*/
void
CrossLinkedRegion::setStartIndex(std::size_t index)
{
  m_startIndex = index;
}

/*!
\brief Returns the start index of this instance.
*/
std::size_t
CrossLinkedRegion::getStartIndex()
{
  return m_startIndex;
}

/*!
\brief Sets the stop index of this instance to \a index.
*/
void
CrossLinkedRegion::setStopIndex(std::size_t index)
{
  m_stopIndex = index;
}

/*!
\brief Returns the stop index of this instance.
*/
std::size_t
CrossLinkedRegion::getStopIndex()
{
  return m_stopIndex;
}


/*!
\brief Returns a const reference to the container of CrossLink instances.
*/
const std::vector<CrossLinkSPtr> &
CrossLinkedRegion::getCrossLinksCstRef() const
{
  return m_crossLinks;
}


/*!
\brief Returns a reference to the container of CrossLink instances.
*/
std::vector<CrossLinkSPtr> &
CrossLinkedRegion::getCrossLinksRef()
{
  return m_crossLinks;
}


/*!
\brief Adds a the \a cross_link_sp CrossLink to the back of the CrossLink container.

Returns the size of the container of CrossLink instances.
*/
std::size_t
CrossLinkedRegion::appendCrossLink(const CrossLinkSPtr &cross_link_sp)
{
  if(cross_link_sp == nullptr || cross_link_sp.get() == nullptr)
    qFatalStream() << "Programming error. Pointer cannot be nullptr.";

  // Only append the cross-link if it is not already in the list.

  std::vector<CrossLinkSPtr>::const_iterator the_iterator_cst =
    std::find_if(m_crossLinks.cbegin(),
                 m_crossLinks.cend(),
                 [&cross_link_sp](const CrossLinkSPtr &iter_cross_link_sp) {
                   return iter_cross_link_sp == cross_link_sp;
                 });

  if(the_iterator_cst == m_crossLinks.cend())
    m_crossLinks.push_back(cross_link_sp);

  return m_crossLinks.size();
}

/*!
\brief Adds the CrossLink item in the \a cross_links container to the back of
the member CrossLink container.

Returns the size of the container of CrossLink instances.
*/
std::size_t
CrossLinkedRegion::appendCrossLinks(
  const std::vector<CrossLinkSPtr> &cross_links)
{
  for(const CrossLinkSPtr &cross_link_sp : cross_links)
    m_crossLinks.push_back(cross_link_sp);

  return m_crossLinks.size();
}

/*!
\brief Removes the \a cross_link_sp CrossLink item from the member CrossLink
container.

Returns the size of the container of CrossLink instances.
*/
std::size_t
CrossLinkedRegion::removeCrossLink(CrossLinkSPtr &cross_link_sp)
{
  std::vector<CrossLinkSPtr>::iterator the_iterator =
    std::find_if(m_crossLinks.begin(),
                 m_crossLinks.end(),
                 [&cross_link_sp](const CrossLinkSPtr &iter_cross_link_sp) {
                   return iter_cross_link_sp == cross_link_sp;
                 });

  if(the_iterator != m_crossLinks.end())
    m_crossLinks.erase(the_iterator);

  return m_crossLinks.size();
}

/*!
\brief Removes the CrossLink at \a index from the member CrossLink
container.

Returns the size of the container of CrossLink instances.
*/
std::size_t
CrossLinkedRegion::removeCrossLinkAt(std::size_t index)
{
  if(index >= m_crossLinks.size())
    qFatalStream() << "Programming error. Index is out of bounds.";

  m_crossLinks.erase(m_crossLinks.begin() + index);

  return m_crossLinks.size();
}

/*!
\brief Assigns \a other to this instance.

The copy of the CrossLink instances is shallow (the shared pointers are copied).
*/
CrossLinkedRegion &
CrossLinkedRegion::operator=(const CrossLinkedRegion &other)
{
  if(this == &other)
    return *this;

  m_startIndex = other.m_startIndex;
  m_stopIndex  = other.m_stopIndex;
  m_crossLinks.assign(other.m_crossLinks.begin(), other.m_crossLinks.end());

  return *this;
}

} // namespace libXpertMassCore
} // namespace MsXpS
