/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various indepstopent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QDebug>
#include <QStringList>
#include <QRegularExpression>
#include <QRegularExpressionMatch>

/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/Utils.hpp"
#include "MsXpS/libXpertMassCore/IndexRangeCollection.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{


/*!
\class MsXpS::libXpertMassCore::IndexRange
\inmodule libXpertMass
\ingroup XpertMassCalculations
\inheaderfile IndexRangeCollection.hpp

\brief The IndexRange structure provides a range of indices that delimit a
region of interest in a  \l{Polymer}  \l{Sequence} instance.

One major IndexRange use case is when defining for what sequence region of a
Polymer the masses or the elemental composition is to be computed. Another use
case is when performing Sequence cleavage or Oligomer fragmentation: the
IndexRange enable the delimitation of the sequence of interest inside a given
Polymer instance.

When instantiating a new IndexRange, the start and stop values are automatically
sorted in ascending order (start <= stop).

\sa IndexRangeCollection, sortAscending(), sortDescending()
*/

/*!
\variable MsXpS::libXpertMassCore::IndexRange::start

\brief Index that marks the start of the index range.
*/

/*!
\variable MsXpS::libXpertMassCore::IndexRange::stop

\brief Index that marks the stop of the index range.
*/

/*!
\fn  MsXpS::libXpertMassCore::IndexRange::IndexRange()

Constructs an IndexRange instance with default
std::numeric_limits<std::size_t>::max() start and stop index ranges.

The use of the max() function to initialize the start and stop member data is to
have a means to check if the IndexRange was initialized by the user or not.
*/

/*!
\overload MsXpS::libXpertMassCore::IndexRange::IndexRange()

\fn IndexRange::IndexRange(std::size_t index_start, std::size_t index_stop)

Constructs an IndexRange instance initializing the start and stop members to the
\a index_start and \a index_stop values, respectively.

The values are sorted in ascending order.
*/

/*!
\overload MsXpS::libXpertMassCore::IndexRange::IndexRange()

\fn  MsXpS::libXpertMassCore::IndexRange::IndexRange(const IndexRange &other)

Constructs an IndexRange instance as a copy of \a other.

Returns a reference to this IndexRange instance.
*/

/*!
\fn  bool MsXpS::libXpertMassCore::IndexRange::operator==(const IndexRange &other)
const

Returns true if this and \a other IndexRange instances are identical.
*/

/*!
\fn  bool MsXpS::libXpertMassCore::IndexRange::operator!=(const IndexRange &other)
const

Returns true if this and \a other IndexRange instances are different.

Returns the negated result of operator==(other).
*/

/*!
\fn  bool MsXpS::libXpertMassCore::IndexRange::sortAscending()

Makes sure that \l start is <= to \l stop.
*/


/*!
\fn  bool MsXpS::libXpertMassCore::IndexRange::sortDescending()

Makes sure that \l start is >= to \l stop.
*/

/*!
\fn QString MsXpS::libXpertMassCore::IndexRange::indicesAsText() const

Returns a string representing this IndexRange instance in the form
"[start-stop]".
*/

/*!
\fn QString MsXpS::libXpertMassCore::IndexRange::positionsAsText() const

Returns a string representing this IndexRange instance in the form
"[start-stop]".

\note The values returned in the string correspond to positions in the Polymer
Sequence,  that is these values correspond to indices incremented by one.
*/

IndexRange::IndexRange(QObject *parent): QObject(parent)
{
}

IndexRange::IndexRange(qsizetype index_start,
                       qsizetype index_stop,
                       QObject *parent)
  : QObject(parent), m_start(index_start), m_stop(index_stop)
{
  sortAscending();
}

IndexRange::IndexRange(const IndexRange &other, QObject *parent)
  : QObject(parent), m_start(other.m_start), m_stop(other.m_stop)
{
}

IndexRange::~IndexRange()
{

}

void
IndexRange::initialize(const IndexRange &other)
{
  m_start = other.m_start;
  m_stop  = other.m_stop;
}

IndexRange *
IndexRange::clone(QObject *parent)
{
  IndexRange *copy_p = new IndexRange(parent);
  copy_p->initialize(*this);
  return copy_p;
}

IndexRange *
IndexRange::clone(const IndexRange &other, QObject *parent)
{
  IndexRange *copy_p = new IndexRange(parent);
  copy_p->initialize(other);
  return copy_p;
}

bool
IndexRange::operator==(const IndexRange &other) const
{
  return m_start == other.m_start && m_stop == other.m_stop;
};

bool
IndexRange::operator!=(const IndexRange &other) const
{
  return !operator==(other);
}

void
IndexRange::sortAscending()
{
  if(m_stop < m_start)
    std::swap(m_start, m_stop);
}

void
IndexRange::sortDescending()
{
  if(m_stop > m_start)
    std::swap(m_start, m_stop);
}

bool
IndexRange::isValid() const
{
  return m_start < std::numeric_limits<qsizetype>::max() &&
         m_stop < std::numeric_limits<qsizetype>::max();
}

void
IndexRange::reset()
{
  m_start = std::numeric_limits<qsizetype>::max();
  m_stop  = std::numeric_limits<qsizetype>::max();
}

QString
IndexRange::indicesAsText() const
{
  return QString("[%1-%2]").arg(m_start).arg(m_stop);
};

QString
IndexRange::positionsAsText() const
{
  return QString("[%1-%2]").arg(m_start + 1).arg(m_stop + 1);
};

void
IndexRange::registerJsConstructor(QJSEngine *engine)

{
  if(!engine)
    {
      qWarning() << "Cannot register IndexRange class: engine is null";
      return;
    }

  // Register the meta object as a constructor

  QJSValue jsMetaObject = engine->newQMetaObject(&IndexRange::staticMetaObject);
  engine->globalObject().setProperty("IndexRange", jsMetaObject);
}


} // namespace libXpertMassCore
} // namespace MsXpS
