/**
 * \file pappsomspp/processing/specpeptidoms/locationsaver.cpp
 * \date 24/03/2025
 * \author Aurélien Berthier
 * \brief save protein subsequences for alignment
 *
 * C++ implementation of the SpecPeptidOMS algorithm described in :
 * (1) Benoist, É.; Jean, G.; Rogniaux, H.; Fertin, G.; Tessier, D. SpecPeptidOMS Directly and
 * Rapidly Aligns Mass Spectra on Whole Proteomes and Identifies Peptides That Are Not Necessarily
 * Tryptic: Implications for Peptidomics. J. Proteome Res. 2025.
 * https://doi.org/10.1021/acs.jproteome.4c00870.
 */

/*
 * Copyright (c) 2025 Aurélien Berthier
 * <aurelien.berthier@ls2n.fr>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <algorithm>
#include "locationsaver.h"
#include "types.h"

pappso::specpeptidoms::LocationSaver::LocationSaver()
{
  m_locations_heap.reserve(MAX_SAVED_ALIGNMENTS);
  for(std::size_t i = 0; i < MAX_SAVED_ALIGNMENTS; i++)
    {
      m_locations_heap.push_back({0, 0, -1, MIN_ALIGNMENT_SCORE, ""});
    }
  // std::make_heap(m_locations_heap.begin(), m_locations_heap.end(),
  // LocationSaver::locationCompare); ?
  // m_tree_scores.reserve ?
}

bool
pappso::specpeptidoms::LocationSaver::locationCompare(const Location &loc1, const Location &loc2)
{
  return loc1.score > loc2.score;
}

// TODO : check complexity if m_tree_in_heap[tree] == true
void
pappso::specpeptidoms::LocationSaver::addLocation(
  std::size_t beginning, std::size_t length, int tree, int score, const QString &protein)
{
  m_tree_scores.at(tree) = score;
  if(m_tree_in_heap.at(tree))
    {
      for(std::vector<Location>::iterator iter = m_locations_heap.begin();
          iter != m_locations_heap.end();
          iter++)
        {
          if(iter->tree == tree)
            {
              iter->score  = score;
              iter->length = length;
            }
        }
      std::make_heap(
        m_locations_heap.begin(), m_locations_heap.end(), LocationSaver::locationCompare);
    }
  else
    {
      if(m_locations_heap.begin()->tree >= 0)
        {
          m_tree_in_heap.at(m_locations_heap.begin()->tree) = false;
        }
      m_tree_in_heap.at(tree) = true;
      std::pop_heap(
        m_locations_heap.begin(), m_locations_heap.end(), LocationSaver::locationCompare);
      m_locations_heap.pop_back();
      m_locations_heap.push_back({beginning, length, tree, score, protein});
      std::push_heap(
        m_locations_heap.begin(), m_locations_heap.end(), LocationSaver::locationCompare);
    }
}

std::vector<pappso::specpeptidoms::Location>
pappso::specpeptidoms::LocationSaver::getLocations() const
{
  std::vector<Location> locations;
  locations.reserve(m_locations_heap.size());
  for(std::vector<Location>::const_iterator iter = m_locations_heap.begin();
      iter != m_locations_heap.end();
      iter++)
    {
      if(iter->tree >= 0)
        {
          locations.push_back(*iter);
        }
    }
  return locations;
}

std::size_t
pappso::specpeptidoms::LocationSaver::getNextTree()
{
  m_tree_scores.push_back(MIN_ALIGNMENT_SCORE);
  m_tree_in_heap.push_back(false);
  return m_tree_scores.size() - 1;
}

int
pappso::specpeptidoms::LocationSaver::getMinScore(int tree_id) const
{
  if(m_tree_scores.size() == 0)
    {
      return m_locations_heap.begin()->score;
    }
  else
    {
      return std::max(m_tree_scores.at(tree_id), m_locations_heap.begin()->score);
    }
}

pappso::specpeptidoms::LocationSaver::~LocationSaver()
{
}

void
pappso::specpeptidoms::LocationSaver::resetLocationSaver()
{
  for(auto iter = m_locations_heap.begin(); iter != m_locations_heap.end(); iter++)
    {
      *(iter) = {0, 0, -1, MIN_ALIGNMENT_SCORE, ""};
    }
  m_tree_scores.clear();
  m_tree_in_heap.clear();
}
