/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Identifiers, createTokenForExternalReference } from '../identifiers';
import { LifecycleHooks } from '../lifecycle_reflector';
import * as o from '../output/output_ast';
import { convertValueToOutputAst } from '../output/value_util';
import { ProviderAstType } from '../template_parser/template_ast';
export function providerDef(ctx, providerAst) {
    let flags = 0 /* None */;
    if (!providerAst.eager) {
        flags |= 4096 /* LazyProvider */;
    }
    if (providerAst.providerType === ProviderAstType.PrivateService) {
        flags |= 8192 /* PrivateProvider */;
    }
    if (providerAst.isModule) {
        flags |= 1073741824 /* TypeModuleProvider */;
    }
    providerAst.lifecycleHooks.forEach((lifecycleHook) => {
        // for regular providers, we only support ngOnDestroy
        if (lifecycleHook === LifecycleHooks.OnDestroy ||
            providerAst.providerType === ProviderAstType.Directive ||
            providerAst.providerType === ProviderAstType.Component) {
            flags |= lifecycleHookToNodeFlag(lifecycleHook);
        }
    });
    const { providerExpr, flags: providerFlags, depsExpr } = providerAst.multiProvider ?
        multiProviderDef(ctx, flags, providerAst.providers) :
        singleProviderDef(ctx, flags, providerAst.providerType, providerAst.providers[0]);
    return {
        providerExpr,
        flags: providerFlags, depsExpr,
        tokenExpr: tokenExpr(ctx, providerAst.token),
    };
}
function multiProviderDef(ctx, flags, providers) {
    const allDepDefs = [];
    const allParams = [];
    const exprs = providers.map((provider, providerIndex) => {
        let expr;
        if (provider.useClass) {
            const depExprs = convertDeps(providerIndex, provider.deps || provider.useClass.diDeps);
            expr = ctx.importExpr(provider.useClass.reference).instantiate(depExprs);
        }
        else if (provider.useFactory) {
            const depExprs = convertDeps(providerIndex, provider.deps || provider.useFactory.diDeps);
            expr = ctx.importExpr(provider.useFactory.reference).callFn(depExprs);
        }
        else if (provider.useExisting) {
            const depExprs = convertDeps(providerIndex, [{ token: provider.useExisting }]);
            expr = depExprs[0];
        }
        else {
            expr = convertValueToOutputAst(ctx, provider.useValue);
        }
        return expr;
    });
    const providerExpr = o.fn(allParams, [new o.ReturnStatement(o.literalArr(exprs))], o.INFERRED_TYPE);
    return {
        providerExpr,
        flags: flags | 1024 /* TypeFactoryProvider */,
        depsExpr: o.literalArr(allDepDefs)
    };
    function convertDeps(providerIndex, deps) {
        return deps.map((dep, depIndex) => {
            const paramName = `p${providerIndex}_${depIndex}`;
            allParams.push(new o.FnParam(paramName, o.DYNAMIC_TYPE));
            allDepDefs.push(depDef(ctx, dep));
            return o.variable(paramName);
        });
    }
}
function singleProviderDef(ctx, flags, providerType, providerMeta) {
    let providerExpr;
    let deps;
    if (providerType === ProviderAstType.Directive || providerType === ProviderAstType.Component) {
        providerExpr = ctx.importExpr(providerMeta.useClass.reference);
        flags |= 16384 /* TypeDirective */;
        deps = providerMeta.deps || providerMeta.useClass.diDeps;
    }
    else {
        if (providerMeta.useClass) {
            providerExpr = ctx.importExpr(providerMeta.useClass.reference);
            flags |= 512 /* TypeClassProvider */;
            deps = providerMeta.deps || providerMeta.useClass.diDeps;
        }
        else if (providerMeta.useFactory) {
            providerExpr = ctx.importExpr(providerMeta.useFactory.reference);
            flags |= 1024 /* TypeFactoryProvider */;
            deps = providerMeta.deps || providerMeta.useFactory.diDeps;
        }
        else if (providerMeta.useExisting) {
            providerExpr = o.NULL_EXPR;
            flags |= 2048 /* TypeUseExistingProvider */;
            deps = [{ token: providerMeta.useExisting }];
        }
        else {
            providerExpr = convertValueToOutputAst(ctx, providerMeta.useValue);
            flags |= 256 /* TypeValueProvider */;
            deps = [];
        }
    }
    const depsExpr = o.literalArr(deps.map(dep => depDef(ctx, dep)));
    return { providerExpr, flags, depsExpr };
}
function tokenExpr(ctx, tokenMeta) {
    return tokenMeta.identifier ? ctx.importExpr(tokenMeta.identifier.reference) :
        o.literal(tokenMeta.value);
}
export function depDef(ctx, dep) {
    // Note: the following fields have already been normalized out by provider_analyzer:
    // - isAttribute, isHost
    const expr = dep.isValue ? convertValueToOutputAst(ctx, dep.value) : tokenExpr(ctx, dep.token);
    let flags = 0 /* None */;
    if (dep.isSkipSelf) {
        flags |= 1 /* SkipSelf */;
    }
    if (dep.isOptional) {
        flags |= 2 /* Optional */;
    }
    if (dep.isSelf) {
        flags |= 4 /* Self */;
    }
    if (dep.isValue) {
        flags |= 8 /* Value */;
    }
    return flags === 0 /* None */ ? expr : o.literalArr([o.literal(flags), expr]);
}
export function lifecycleHookToNodeFlag(lifecycleHook) {
    let nodeFlag = 0 /* None */;
    switch (lifecycleHook) {
        case LifecycleHooks.AfterContentChecked:
            nodeFlag = 2097152 /* AfterContentChecked */;
            break;
        case LifecycleHooks.AfterContentInit:
            nodeFlag = 1048576 /* AfterContentInit */;
            break;
        case LifecycleHooks.AfterViewChecked:
            nodeFlag = 8388608 /* AfterViewChecked */;
            break;
        case LifecycleHooks.AfterViewInit:
            nodeFlag = 4194304 /* AfterViewInit */;
            break;
        case LifecycleHooks.DoCheck:
            nodeFlag = 262144 /* DoCheck */;
            break;
        case LifecycleHooks.OnChanges:
            nodeFlag = 524288 /* OnChanges */;
            break;
        case LifecycleHooks.OnDestroy:
            nodeFlag = 131072 /* OnDestroy */;
            break;
        case LifecycleHooks.OnInit:
            nodeFlag = 65536 /* OnInit */;
            break;
    }
    return nodeFlag;
}
export function componentFactoryResolverProviderDef(reflector, ctx, flags, entryComponents) {
    const entryComponentFactories = entryComponents.map((entryComponent) => ctx.importExpr(entryComponent.componentFactory));
    const token = createTokenForExternalReference(reflector, Identifiers.ComponentFactoryResolver);
    const classMeta = {
        diDeps: [
            { isValue: true, value: o.literalArr(entryComponentFactories) },
            { token: token, isSkipSelf: true, isOptional: true },
            { token: createTokenForExternalReference(reflector, Identifiers.NgModuleRef) },
        ],
        lifecycleHooks: [],
        reference: reflector.resolveExternalReference(Identifiers.CodegenComponentFactoryResolver)
    };
    const { providerExpr, flags: providerFlags, depsExpr } = singleProviderDef(ctx, flags, ProviderAstType.PrivateService, {
        token,
        multi: false,
        useClass: classMeta,
    });
    return { providerExpr, flags: providerFlags, depsExpr, tokenExpr: tokenExpr(ctx, token) };
}
//# sourceMappingURL=data:application/json;base64,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