/*******************************************************************************
* Copyright 2021-2022 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP API oneapi::mkl::sparse::sort_matrix to perform general
*       sparse matrix sorting on a SYCL device (CPU, GPU).
*
*       The input matrix is sorted in-place and no temporary workspace is used.
*
*       The supported floating point data types for sort_matrix matrix data are:
*           float
*           double
*
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <CL/sycl.hpp>

// local includes
#include "../common/common_for_examples.hpp"
#include "common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix sorting
//
template <typename fp, typename intType>
int run_sparse_matrix_sorting_example(const sycl::device &dev)
{
    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::gemv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // create execution main_queue
    sycl::queue main_queue(dev, exception_handler);
    sycl::context cxt = main_queue.get_context();

    // Initialize data for Sparse Matrix
    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::one;
    intType size = 4;
    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;

    //
    // setup A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fp, mkl_allocator<fp, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(27 * a_nrows);
    a.resize(27 * a_nrows);

    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fp, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;
    shuffle_matrix_data<fp, intType>(ia, ja, a, a_ind, a_nrows, a_nnz);

    //
    // Execute Matrix Sorting
    //

    const fp ave_nnz_per_row = static_cast<fp>(a_nnz)/a_nrows;

    intType max_nnz_per_row = 0;
    intType min_nnz_per_row = a_nnz;
    {
        for (intType row = 0; row < a_nrows; ++row) {
            const intType loc_nnz_per_row = ia[row+1] - ia[row];

            if (loc_nnz_per_row > max_nnz_per_row) 
                max_nnz_per_row = loc_nnz_per_row;

            if (loc_nnz_per_row < min_nnz_per_row) 
                min_nnz_per_row = loc_nnz_per_row;
        }
    }

    std::cout << "\n\t\tsparse::sort_matrix parameters:\n";
    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tA_nnz   = " << a_nnz << std::endl;
    std::cout << "\t\t\t\tmin_nnz_per_row = " << min_nnz_per_row << std::endl;
    std::cout << "\t\t\t\tave_nnz_per_row = " << ave_nnz_per_row << std::endl;
    std::cout << "\t\t\t\tmax_nnz_per_row = " << max_nnz_per_row << std::endl;

    oneapi::mkl::sparse::matrix_handle_t A = nullptr;

    sycl::buffer<intType, 1> a_rowptr(ia.data(), a_nrows + 1);
    sycl::buffer<intType, 1> a_colind(ja.data(), a_nnz);
    sycl::buffer<fp, 1> a_val(a.data(), a_nnz);

    // Print part of unsorted matrix input
    {
        auto ia       = a_rowptr.get_host_access(sycl::read_only);
        auto ja       = a_colind.get_host_access(sycl::read_only);
        auto a        = a_val.get_host_access(sycl::read_only);
        std::cout << "[INPUT] Unsorted A matrix [first two rows]:" << std::endl;
        for (intType row = 0; row < std::min(static_cast<intType>(2), a_nrows); ++row) {
            for (intType j = ia[row] - a_ind; j < ia[row + 1] - a_ind; ++j) {
                intType col = ja[j];
                fp val  = a[j];
                std::cout << "A(" << row + a_ind << ", " << col << ") = " << val << std::endl;
            }
        }
    }

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);

        oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols, a_index, a_rowptr, a_colind, a_val);

        // Sort the sparse matrix
        oneapi::mkl::sparse::sort_matrix(main_queue, A);

        // Print part of sorted matrix output
        {
            auto ia       = a_rowptr.get_host_access(sycl::read_only);
            auto ja       = a_colind.get_host_access(sycl::read_only);
            auto a        = a_val.get_host_access(sycl::read_only);
            std::cout << "[OUTPUT] Sorted A matrix [first two rows]:" << std::endl;
            for (intType row = 0; row < std::min(static_cast<intType>(2), a_nrows); ++row) {
                for (intType j = ia[row] - a_ind; j < ia[row + 1] - a_ind; ++j) {
                    intType col = ja[j];
                    fp val  = a[j];
                    std::cout << "A(" << row + a_ind << ", " << col << ") = " << val << std::endl;
                }
            }
        }

        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A);
        return 1;
    }

    main_queue.wait();

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# CSR Matrix Sorting Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::sort_matrix" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_sorting_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_sorting_example<double, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    return status;
}
