# The contents of this file are automatically written by
# tools/generate_schema_wrapper.py. Do not modify directly.

from __future__ import annotations

# These errors need to be ignored as they come from the overload methods
# which trigger two kind of errors in mypy:
# * all of them do not have an implementation in this file
# * some of them are the only overload methods -> overloads usually only make
#   sense if there are multiple ones
# However, we need these overloads due to how the propertysetter works
# mypy: disable-error-code="no-overload-impl, empty-body, misc"
import sys
from typing import TYPE_CHECKING, Any, Literal, TypedDict, Union, overload

if sys.version_info >= (3, 10):
    from typing import TypeAlias
else:
    from typing_extensions import TypeAlias
import narwhals.stable.v1 as nw

from altair.utils import infer_encoding_types as _infer_encoding_types
from altair.utils import parse_shorthand
from altair.utils.schemapi import Undefined, with_property_setters

from . import core
from ._typing import *  # noqa: F403

if TYPE_CHECKING:
    # ruff: noqa: F405
    from collections.abc import Sequence

    from altair import Parameter, SchemaBase
    from altair.typing import Optional
    from altair.vegalite.v5.api import Bin, Impute, IntoCondition
    from altair.vegalite.v5.schema.core import (
        Axis,
        DateTime,
        EncodingSortField,
        Header,
        Legend,
        RepeatRef,
        Scale,
        TimeUnitParams,
    )

    if sys.version_info >= (3, 11):
        from typing import Self
    else:
        from typing_extensions import Self


__all__ = [
    "X2",
    "Y2",
    "Angle",
    "AngleDatum",
    "AngleValue",
    "Color",
    "ColorDatum",
    "ColorValue",
    "Column",
    "DatumChannelMixin",
    "Description",
    "DescriptionValue",
    "Detail",
    "Facet",
    "FieldChannelMixin",
    "Fill",
    "FillDatum",
    "FillOpacity",
    "FillOpacityDatum",
    "FillOpacityValue",
    "FillValue",
    "Href",
    "HrefValue",
    "Key",
    "Latitude",
    "Latitude2",
    "Latitude2Datum",
    "Latitude2Value",
    "LatitudeDatum",
    "Longitude",
    "Longitude2",
    "Longitude2Datum",
    "Longitude2Value",
    "LongitudeDatum",
    "Opacity",
    "OpacityDatum",
    "OpacityValue",
    "Order",
    "OrderValue",
    "Radius",
    "Radius2",
    "Radius2Datum",
    "Radius2Value",
    "RadiusDatum",
    "RadiusValue",
    "Row",
    "Shape",
    "ShapeDatum",
    "ShapeValue",
    "Size",
    "SizeDatum",
    "SizeValue",
    "Stroke",
    "StrokeDash",
    "StrokeDashDatum",
    "StrokeDashValue",
    "StrokeDatum",
    "StrokeOpacity",
    "StrokeOpacityDatum",
    "StrokeOpacityValue",
    "StrokeValue",
    "StrokeWidth",
    "StrokeWidthDatum",
    "StrokeWidthValue",
    "Text",
    "TextDatum",
    "TextValue",
    "Theta",
    "Theta2",
    "Theta2Datum",
    "Theta2Value",
    "ThetaDatum",
    "ThetaValue",
    "Tooltip",
    "TooltipValue",
    "Url",
    "UrlValue",
    "ValueChannelMixin",
    "X",
    "X2Datum",
    "X2Value",
    "XDatum",
    "XError",
    "XError2",
    "XError2Value",
    "XErrorValue",
    "XOffset",
    "XOffsetDatum",
    "XOffsetValue",
    "XValue",
    "Y",
    "Y2Datum",
    "Y2Value",
    "YDatum",
    "YError",
    "YError2",
    "YError2Value",
    "YErrorValue",
    "YOffset",
    "YOffsetDatum",
    "YOffsetValue",
    "YValue",
    "with_property_setters",
]


class FieldChannelMixin:
    _encoding_name: str

    def to_dict(
        self,
        validate: bool = True,
        ignore: list[str] | None = None,
        context: dict[str, Any] | None = None,
    ) -> dict | list[dict]:
        context = context or {}
        ignore = ignore or []
        shorthand = self._get("shorthand")  # type: ignore[attr-defined]
        field = self._get("field")  # type: ignore[attr-defined]

        if shorthand is not Undefined and field is not Undefined:
            msg = f"{self.__class__.__name__} specifies both shorthand={shorthand} and field={field}. "
            raise ValueError(msg)

        if isinstance(shorthand, (tuple, list)):
            # If given a list of shorthands, then transform it to a list of classes
            kwds = self._kwds.copy()  # type: ignore[attr-defined]
            kwds.pop("shorthand")
            return [
                self.__class__(sh, **kwds).to_dict(  # type: ignore[call-arg]
                    validate=validate, ignore=ignore, context=context
                )
                for sh in shorthand
            ]

        if shorthand is Undefined:
            parsed = {}
        elif isinstance(shorthand, str):
            data: nw.DataFrame | Any = context.get("data", None)
            parsed = parse_shorthand(shorthand, data=data)
            type_required = "type" in self._kwds  # type: ignore[attr-defined]
            type_in_shorthand = "type" in parsed
            type_defined_explicitly = self._get("type") is not Undefined  # type: ignore[attr-defined]
            if not type_required:
                # Secondary field names don't require a type argument in VegaLite 3+.
                # We still parse it out of the shorthand, but drop it here.
                parsed.pop("type", None)
            elif not (type_in_shorthand or type_defined_explicitly):
                if isinstance(data, nw.DataFrame):
                    msg = (
                        f'Unable to determine data type for the field "{shorthand}";'
                        " verify that the field name is not misspelled."
                        " If you are referencing a field from a transform,"
                        " also confirm that the data type is specified correctly."
                    )
                    raise ValueError(msg)
                else:
                    msg = (
                        f"{shorthand} encoding field is specified without a type; "
                        "the type cannot be automatically inferred because "
                        "the data is not specified as a pandas.DataFrame."
                    )
                    raise ValueError(msg)
        else:
            # Shorthand is not a string; we pass the definition to field,
            # and do not do any parsing.
            parsed = {"field": shorthand}
        context["parsed_shorthand"] = parsed

        return super().to_dict(validate=validate, ignore=ignore, context=context)


class ValueChannelMixin:
    _encoding_name: str

    def to_dict(
        self,
        validate: bool = True,
        ignore: list[str] | None = None,
        context: dict[str, Any] | None = None,
    ) -> dict:
        context = context or {}
        ignore = ignore or []
        condition = self._get("condition", Undefined)  # type: ignore[attr-defined]
        copy = self  # don't copy unless we need to
        if condition is not Undefined:
            if isinstance(condition, core.SchemaBase):
                pass
            elif "field" in condition and "type" not in condition:
                kwds = parse_shorthand(condition["field"], context.get("data", None))
                copy = self.copy(deep=["condition"])  # type: ignore[attr-defined]
                copy["condition"].update(kwds)  # type: ignore[index]
        return super(ValueChannelMixin, copy).to_dict(
            validate=validate, ignore=ignore, context=context
        )


class DatumChannelMixin:
    _encoding_name: str

    def to_dict(
        self,
        validate: bool = True,
        ignore: list[str] | None = None,
        context: dict[str, Any] | None = None,
    ) -> dict:
        context = context or {}
        ignore = ignore or []
        datum = self._get("datum", Undefined)  # type: ignore[attr-defined] # noqa
        copy = self  # don't copy unless we need to
        return super(DatumChannelMixin, copy).to_dict(
            validate=validate, ignore=ignore, context=context
        )


@with_property_setters
class Angle(FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber):
    r"""
    Angle schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "angle"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Angle: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Angle: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Angle: ...
    @overload
    def bandPosition(self, _: float, /) -> Angle: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Angle: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Angle: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Angle: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Angle: ...
    @overload
    def condition(self, _: list[core.ConditionalValueDefnumberExprRef], /) -> Angle: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Angle: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Angle: ...
    @overload
    def legend(self, _: Legend | None, /) -> Angle: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Angle: ...
    @overload
    def scale(self, _: Scale | None, /) -> Angle: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Angle: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Angle: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Angle: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Angle: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Angle: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Angle: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Angle: ...
    @overload
    def type(self, _: StandardType_T, /) -> Angle: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class AngleDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber):
    """
    AngleDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "angle"

    @overload
    def bandPosition(self, _: float, /) -> AngleDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> AngleDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> AngleDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> AngleDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> AngleDatum: ...
    @overload
    def type(self, _: Type_T, /) -> AngleDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class AngleValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    AngleValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "angle"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> AngleValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> AngleValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> AngleValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> AngleValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> AngleValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> AngleValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> AngleValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Color(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefGradientstringnull,
):
    r"""
    Color schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "color"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Color: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Color: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Color: ...
    @overload
    def bandPosition(self, _: float, /) -> Color: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Color: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Color: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Color: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Color: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> Color: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Color: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Color: ...
    @overload
    def legend(self, _: Legend | None, /) -> Color: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Color: ...
    @overload
    def scale(self, _: Scale | None, /) -> Color: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Color: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Color: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Color: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Color: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Color: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Color: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Color: ...
    @overload
    def type(self, _: StandardType_T, /) -> Color: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ColorDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefGradientstringnull
):
    """
    ColorDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "color"

    @overload
    def bandPosition(self, _: float, /) -> ColorDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ColorDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ColorDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> ColorDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> ColorDatum: ...
    @overload
    def type(self, _: Type_T, /) -> ColorDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ColorValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefGradientstringnull,
):
    """
    ColorValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, :class:`Gradient`, :class:`LinearGradient`, :class:`RadialGradient`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "color"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> ColorValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> ColorValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> ColorValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> ColorValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ColorValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ColorValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> ColorValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Column(FieldChannelMixin, core.RowColumnEncodingFieldDef):
    r"""
    Column schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    align : :class:`LayoutAlign`, Literal['all', 'each', 'none']
        The alignment to apply to row/column facet's subplot. The supported string values
        are ``"all"``, ``"each"``, and ``"none"``.

        * For ``"none"``, a flow layout will be used, in which adjacent subviews are simply
          placed one after the other.
        * For ``"each"``, subviews will be aligned into a clean grid structure, but each row
          or column may be of variable size.
        * For ``"all"``, subviews will be aligned and each row or column will be sized
          identically based on the maximum observed size. String values for this property
          will be applied to both grid rows and columns.

        **Default value:** ``"all"``.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    center : bool
        Boolean flag indicating if facet's subviews should be centered relative to their
        respective rows or columns.

        **Default value:** ``false``
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    header : dict, :class:`Header`, None
        An object defining properties of a facet's header.
    sort : dict, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`EncodingSortField`, Sequence[dict, :class:`DateTime`], Literal['ascending', 'descending'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` is not supported for ``row`` and ``column``.
    spacing : float
        The spacing in pixels between facet's sub-views.

        **Default value**: Depends on ``"spacing"`` property of `the view composition
        configuration <https://vega.github.io/vega-lite/docs/config.html#view-config>`__
        (``20`` by default)
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "column"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Column: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Column: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Column: ...
    @overload
    def align(self, _: LayoutAlign_T, /) -> Column: ...
    @overload
    def bandPosition(self, _: float, /) -> Column: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Column: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Column: ...
    @overload
    def center(self, _: bool, /) -> Column: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Column: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Column: ...
    @overload
    def header(self, _: Header | None, /) -> Column: ...
    @overload
    def header(
        self,
        *,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        labelAngle: Optional[float] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOrient: Optional[SchemaBase | Orient_T] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labels: Optional[bool] = Undefined,
        orient: Optional[SchemaBase | Orient_T] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Column: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | EncodingSortField
        | Sequence[DateTime | Temporal]
        | SortOrder_T
        | None,
        /,
    ) -> Column: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Column: ...
    @overload
    def spacing(self, _: float, /) -> Column: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Column: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Column: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Column: ...
    @overload
    def type(self, _: StandardType_T, /) -> Column: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        align: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        center: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        header: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | SortOrder_T
            | None
        ] = Undefined,
        spacing: Optional[float] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            align=align,
            bandPosition=bandPosition,
            bin=bin,
            center=center,
            field=field,
            header=header,
            sort=sort,
            spacing=spacing,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Description(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Description schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "description"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Description: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Description: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Description: ...
    @overload
    def bandPosition(self, _: float, /) -> Description: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Description: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Description: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Description: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Description: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringExprRef], /
    ) -> Description: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Description: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Description: ...
    @overload
    def format(self, _: str, /) -> Description: ...
    @overload
    def format(self, _: Map, /) -> Description: ...
    @overload
    def formatType(self, _: str, /) -> Description: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Description: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Description: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Description: ...
    @overload
    def type(self, _: StandardType_T, /) -> Description: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class DescriptionValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    DescriptionValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "description"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> DescriptionValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> DescriptionValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> DescriptionValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> DescriptionValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> DescriptionValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> DescriptionValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> DescriptionValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Detail(FieldChannelMixin, core.FieldDefWithoutScale):
    r"""
    Detail schema wrapper.

    Definition object for a data field, its type and transformation of an encoding channel.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "detail"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Detail: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Detail: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Detail: ...
    @overload
    def bandPosition(self, _: float, /) -> Detail: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Detail: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Detail: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Detail: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Detail: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Detail: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Detail: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Detail: ...
    @overload
    def type(self, _: StandardType_T, /) -> Detail: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Facet(FieldChannelMixin, core.FacetEncodingFieldDef):
    r"""
    Facet schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    align : dict, :class:`LayoutAlign`, :class:`RowColLayoutAlign`, Literal['all', 'each', 'none']
        The alignment to apply to grid rows and columns. The supported string values are
        ``"all"``, ``"each"``, and ``"none"``.

        * For ``"none"``, a flow layout will be used, in which adjacent subviews are simply
          placed one after the other.
        * For ``"each"``, subviews will be aligned into a clean grid structure, but each row
          or column may be of variable size.
        * For ``"all"``, subviews will be aligned and each row or column will be sized
          identically based on the maximum observed size. String values for this property
          will be applied to both grid rows and columns.

        Alternatively, an object value of the form ``{"row": string, "column": string}`` can
        be used to supply different alignments for rows and columns.

        **Default value:** ``"all"``.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    bounds : Literal['full', 'flush']
        The bounds calculation method to use for determining the extent of a sub-plot. One
        of ``full`` (the default) or ``flush``.

        * If set to ``full``, the entire calculated bounds (including axes, title, and
          legend) will be used.
        * If set to ``flush``, only the specified width and height values for the sub-view
          will be used. The ``flush`` setting can be useful when attempting to place
          sub-plots without axes or legends into a uniform grid structure.

        **Default value:** ``"full"``
    center : bool, dict, :class:`RowColboolean`
        Boolean flag indicating if subviews should be centered relative to their respective
        rows or columns.

        An object value of the form ``{"row": boolean, "column": boolean}`` can be used to
        supply different centering values for rows and columns.

        **Default value:** ``false``
    columns : float
        The number of columns to include in the view composition layout.

        **Default value**: ``undefined`` -- An infinite number of columns (a single row)
        will be assumed. This is equivalent to ``hconcat`` (for ``concat``) and to using the
        ``column`` channel (for ``facet`` and ``repeat``).

        **Note**:

        1) This property is only for:

        * the general (wrappable) ``concat`` operator (not ``hconcat``/``vconcat``)
        * the ``facet`` and ``repeat`` operator with one field/repetition definition
          (without row/column nesting)

        2) Setting the ``columns`` to ``1`` is equivalent to ``vconcat`` (for ``concat``)
        and to using the ``row`` channel (for ``facet`` and ``repeat``).
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    header : dict, :class:`Header`, None
        An object defining properties of a facet's header.
    sort : dict, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`EncodingSortField`, Sequence[dict, :class:`DateTime`], Literal['ascending', 'descending'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` is not supported for ``row`` and ``column``.
    spacing : dict, float, :class:`RowColnumber`
        The spacing in pixels between sub-views of the composition operator. An object of
        the form ``{"row": number, "column": number}`` can be used to set different spacing
        values for rows and columns.

        **Default value**: Depends on ``"spacing"`` property of `the view composition
        configuration <https://vega.github.io/vega-lite/docs/config.html#view-config>`__
        (``20`` by default)
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "facet"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Facet: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Facet: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Facet: ...
    @overload
    def align(self, _: RowColKwds[LayoutAlign_T] | LayoutAlign_T, /) -> Facet: ...
    @overload
    def align(
        self,
        *,
        column: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        row: Optional[SchemaBase | LayoutAlign_T] = Undefined,
    ) -> Facet: ...
    @overload
    def bandPosition(self, _: float, /) -> Facet: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Facet: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Facet: ...
    @overload
    def bounds(self, _: Literal["full", "flush"], /) -> Facet: ...
    @overload
    def center(self, _: bool | RowColKwds[bool], /) -> Facet: ...
    @overload
    def center(
        self, *, column: Optional[bool] = Undefined, row: Optional[bool] = Undefined
    ) -> Facet: ...
    @overload
    def columns(self, _: float, /) -> Facet: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Facet: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Facet: ...
    @overload
    def header(self, _: Header | None, /) -> Facet: ...
    @overload
    def header(
        self,
        *,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        labelAngle: Optional[float] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOrient: Optional[SchemaBase | Orient_T] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labels: Optional[bool] = Undefined,
        orient: Optional[SchemaBase | Orient_T] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Facet: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | EncodingSortField
        | Sequence[DateTime | Temporal]
        | SortOrder_T
        | None,
        /,
    ) -> Facet: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Facet: ...
    @overload
    def spacing(self, _: float | RowColKwds[float], /) -> Facet: ...
    @overload
    def spacing(
        self, *, column: Optional[float] = Undefined, row: Optional[float] = Undefined
    ) -> Facet: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Facet: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Facet: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Facet: ...
    @overload
    def type(self, _: StandardType_T, /) -> Facet: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        align: Optional[SchemaBase | Map | LayoutAlign_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        bounds: Optional[Literal["full", "flush"]] = Undefined,
        center: Optional[bool | SchemaBase | Map] = Undefined,
        columns: Optional[float] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        header: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | SortOrder_T
            | None
        ] = Undefined,
        spacing: Optional[float | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            align=align,
            bandPosition=bandPosition,
            bin=bin,
            bounds=bounds,
            center=center,
            columns=columns,
            field=field,
            header=header,
            sort=sort,
            spacing=spacing,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Fill(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefGradientstringnull,
):
    r"""
    Fill schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fill"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Fill: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Fill: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Fill: ...
    @overload
    def bandPosition(self, _: float, /) -> Fill: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Fill: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Fill: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Fill: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Fill: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> Fill: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Fill: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Fill: ...
    @overload
    def legend(self, _: Legend | None, /) -> Fill: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Fill: ...
    @overload
    def scale(self, _: Scale | None, /) -> Fill: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Fill: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Fill: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Fill: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Fill: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Fill: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Fill: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Fill: ...
    @overload
    def type(self, _: StandardType_T, /) -> Fill: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefGradientstringnull
):
    """
    FillDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fill"

    @overload
    def bandPosition(self, _: float, /) -> FillDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> FillDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> FillDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> FillDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> FillDatum: ...
    @overload
    def type(self, _: Type_T, /) -> FillDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefGradientstringnull,
):
    """
    FillValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, :class:`Gradient`, :class:`LinearGradient`, :class:`RadialGradient`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fill"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> FillValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> FillValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> FillValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> FillValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> FillValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> FillValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> FillValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class FillOpacity(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    FillOpacity schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fillOpacity"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> FillOpacity: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> FillOpacity: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> FillOpacity: ...
    @overload
    def bandPosition(self, _: float, /) -> FillOpacity: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> FillOpacity: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> FillOpacity: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> FillOpacity: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def legend(self, _: Legend | None, /) -> FillOpacity: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def scale(self, _: Scale | None, /) -> FillOpacity: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> FillOpacity: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> FillOpacity: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> FillOpacity: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> FillOpacity: ...
    @overload
    def type(self, _: StandardType_T, /) -> FillOpacity: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillOpacityDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber
):
    """
    FillOpacityDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fillOpacity"

    @overload
    def bandPosition(self, _: float, /) -> FillOpacityDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacityDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacityDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> FillOpacityDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> FillOpacityDatum: ...
    @overload
    def type(self, _: Type_T, /) -> FillOpacityDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class FillOpacityValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    FillOpacityValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "fillOpacity"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> FillOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> FillOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> FillOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> FillOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> FillOpacityValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> FillOpacityValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Href(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Href schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "href"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Href: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Href: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Href: ...
    @overload
    def bandPosition(self, _: float, /) -> Href: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Href: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Href: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Href: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Href: ...
    @overload
    def condition(self, _: list[core.ConditionalValueDefstringExprRef], /) -> Href: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Href: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Href: ...
    @overload
    def format(self, _: str, /) -> Href: ...
    @overload
    def format(self, _: Map, /) -> Href: ...
    @overload
    def formatType(self, _: str, /) -> Href: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Href: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Href: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Href: ...
    @overload
    def type(self, _: StandardType_T, /) -> Href: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class HrefValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    HrefValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "href"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> HrefValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> HrefValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> HrefValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> HrefValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> HrefValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> HrefValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> HrefValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Key(FieldChannelMixin, core.FieldDefWithoutScale):
    r"""
    Key schema wrapper.

    Definition object for a data field, its type and transformation of an encoding channel.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "key"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Key: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Key: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Key: ...
    @overload
    def bandPosition(self, _: float, /) -> Key: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Key: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Key: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Key: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Key: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Key: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Key: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Key: ...
    @overload
    def type(self, _: StandardType_T, /) -> Key: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Latitude(FieldChannelMixin, core.LatLongFieldDef):
    r"""
    Latitude schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : Literal['quantitative']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Latitude: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Latitude: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Latitude: ...
    @overload
    def bandPosition(self, _: float, /) -> Latitude: ...
    @overload
    def bin(self, _: None, /) -> Latitude: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Latitude: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Latitude: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Latitude: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Latitude: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Latitude: ...
    @overload
    def type(self, _: Literal["quantitative"], /) -> Latitude: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[Literal["quantitative"]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class LatitudeDatum(DatumChannelMixin, core.DatumDef):
    """
    LatitudeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude"

    @overload
    def bandPosition(self, _: float, /) -> LatitudeDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> LatitudeDatum: ...
    @overload
    def type(self, _: Type_T, /) -> LatitudeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Latitude2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Latitude2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Latitude2: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Latitude2: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Latitude2: ...
    @overload
    def bandPosition(self, _: float, /) -> Latitude2: ...
    @overload
    def bin(self, _: None, /) -> Latitude2: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Latitude2: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Latitude2: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Latitude2: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Latitude2: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Latitude2: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Latitude2Datum(DatumChannelMixin, core.DatumDef):
    """
    Latitude2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude2"

    @overload
    def bandPosition(self, _: float, /) -> Latitude2Datum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Latitude2Datum: ...
    @overload
    def type(self, _: Type_T, /) -> Latitude2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Latitude2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Latitude2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "latitude2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Longitude(FieldChannelMixin, core.LatLongFieldDef):
    r"""
    Longitude schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : Literal['quantitative']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Longitude: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Longitude: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Longitude: ...
    @overload
    def bandPosition(self, _: float, /) -> Longitude: ...
    @overload
    def bin(self, _: None, /) -> Longitude: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Longitude: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Longitude: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Longitude: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Longitude: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Longitude: ...
    @overload
    def type(self, _: Literal["quantitative"], /) -> Longitude: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[Literal["quantitative"]] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class LongitudeDatum(DatumChannelMixin, core.DatumDef):
    """
    LongitudeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude"

    @overload
    def bandPosition(self, _: float, /) -> LongitudeDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> LongitudeDatum: ...
    @overload
    def type(self, _: Type_T, /) -> LongitudeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Longitude2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Longitude2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Longitude2: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Longitude2: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Longitude2: ...
    @overload
    def bandPosition(self, _: float, /) -> Longitude2: ...
    @overload
    def bin(self, _: None, /) -> Longitude2: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Longitude2: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Longitude2: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Longitude2: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Longitude2: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Longitude2: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Longitude2Datum(DatumChannelMixin, core.DatumDef):
    """
    Longitude2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude2"

    @overload
    def bandPosition(self, _: float, /) -> Longitude2Datum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Longitude2Datum: ...
    @overload
    def type(self, _: Type_T, /) -> Longitude2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Longitude2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Longitude2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "longitude2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Opacity(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    Opacity schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "opacity"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Opacity: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Opacity: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Opacity: ...
    @overload
    def bandPosition(self, _: float, /) -> Opacity: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Opacity: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Opacity: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Opacity: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Opacity: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> Opacity: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Opacity: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Opacity: ...
    @overload
    def legend(self, _: Legend | None, /) -> Opacity: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Opacity: ...
    @overload
    def scale(self, _: Scale | None, /) -> Opacity: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Opacity: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Opacity: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Opacity: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Opacity: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Opacity: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Opacity: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Opacity: ...
    @overload
    def type(self, _: StandardType_T, /) -> Opacity: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class OpacityDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber):
    """
    OpacityDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "opacity"

    @overload
    def bandPosition(self, _: float, /) -> OpacityDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> OpacityDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> OpacityDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> OpacityDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> OpacityDatum: ...
    @overload
    def type(self, _: Type_T, /) -> OpacityDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class OpacityValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    OpacityValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "opacity"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> OpacityValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> OpacityValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> OpacityValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> OpacityValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> OpacityValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> OpacityValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> OpacityValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Order(FieldChannelMixin, core.OrderFieldDef):
    r"""
    Order schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    sort : :class:`SortOrder`, Literal['ascending', 'descending']
        The sort order. One of ``"ascending"`` (default) or ``"descending"``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "order"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Order: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Order: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Order: ...
    @overload
    def bandPosition(self, _: float, /) -> Order: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Order: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Order: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Order: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Order: ...
    @overload
    def sort(self, _: SortOrder_T, /) -> Order: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Order: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Order: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Order: ...
    @overload
    def type(self, _: StandardType_T, /) -> Order: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        sort: Optional[SchemaBase | SortOrder_T] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class OrderValue(ValueChannelMixin, core.OrderValueDef):
    """
    OrderValue schema wrapper.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    condition : dict, :class:`ConditionalValueDefnumber`, :class:`ConditionalParameterValueDefnumber`, :class:`ConditionalPredicateValueDefnumber`, Sequence[dict, :class:`ConditionalValueDefnumber`, :class:`ConditionalParameterValueDefnumber`, :class:`ConditionalPredicateValueDefnumber`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "order"

    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float] = Undefined,
    ) -> OrderValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float] = Undefined,
    ) -> OrderValue: ...
    @overload
    def condition(self, _: list[core.ConditionalValueDefnumber], /) -> OrderValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Radius(FieldChannelMixin, core.PositionFieldDefBase):
    r"""
    Radius schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    stack : bool, :class:`StackOffset`, Literal['zero', 'center', 'normalize'], None
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).
        * ``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Radius: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Radius: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Radius: ...
    @overload
    def bandPosition(self, _: float, /) -> Radius: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Radius: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Radius: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Radius: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Radius: ...
    @overload
    def scale(self, _: Scale | None, /) -> Radius: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Radius: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Radius: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Radius: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Radius: ...
    @overload
    def stack(self, _: bool | StackOffset_T | None, /) -> Radius: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Radius: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Radius: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Radius: ...
    @overload
    def type(self, _: StandardType_T, /) -> Radius: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        stack: Optional[bool | SchemaBase | StackOffset_T | None] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            scale=scale,
            sort=sort,
            stack=stack,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class RadiusDatum(DatumChannelMixin, core.PositionDatumDefBase):
    """
    RadiusDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    stack : bool, :class:`StackOffset`, Literal['zero', 'center', 'normalize'], None
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).
        * ``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius"

    @overload
    def bandPosition(self, _: float, /) -> RadiusDatum: ...
    @overload
    def scale(self, _: Scale | None, /) -> RadiusDatum: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> RadiusDatum: ...
    @overload
    def stack(self, _: bool | StackOffset_T | None, /) -> RadiusDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> RadiusDatum: ...
    @overload
    def type(self, _: Type_T, /) -> RadiusDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        stack: Optional[bool | SchemaBase | StackOffset_T | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            scale=scale,
            stack=stack,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class RadiusValue(ValueChannelMixin, core.PositionValueDef):
    """
    RadiusValue schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Radius2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Radius2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Radius2: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Radius2: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Radius2: ...
    @overload
    def bandPosition(self, _: float, /) -> Radius2: ...
    @overload
    def bin(self, _: None, /) -> Radius2: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Radius2: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Radius2: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Radius2: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Radius2: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Radius2: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Radius2Datum(DatumChannelMixin, core.DatumDef):
    """
    Radius2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius2"

    @overload
    def bandPosition(self, _: float, /) -> Radius2Datum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Radius2Datum: ...
    @overload
    def type(self, _: Type_T, /) -> Radius2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Radius2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Radius2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "radius2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Row(FieldChannelMixin, core.RowColumnEncodingFieldDef):
    r"""
    Row schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    align : :class:`LayoutAlign`, Literal['all', 'each', 'none']
        The alignment to apply to row/column facet's subplot. The supported string values
        are ``"all"``, ``"each"``, and ``"none"``.

        * For ``"none"``, a flow layout will be used, in which adjacent subviews are simply
          placed one after the other.
        * For ``"each"``, subviews will be aligned into a clean grid structure, but each row
          or column may be of variable size.
        * For ``"all"``, subviews will be aligned and each row or column will be sized
          identically based on the maximum observed size. String values for this property
          will be applied to both grid rows and columns.

        **Default value:** ``"all"``.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    center : bool
        Boolean flag indicating if facet's subviews should be centered relative to their
        respective rows or columns.

        **Default value:** ``false``
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    header : dict, :class:`Header`, None
        An object defining properties of a facet's header.
    sort : dict, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`EncodingSortField`, Sequence[dict, :class:`DateTime`], Literal['ascending', 'descending'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` is not supported for ``row`` and ``column``.
    spacing : float
        The spacing in pixels between facet's sub-views.

        **Default value**: Depends on ``"spacing"`` property of `the view composition
        configuration <https://vega.github.io/vega-lite/docs/config.html#view-config>`__
        (``20`` by default)
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "row"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Row: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Row: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Row: ...
    @overload
    def align(self, _: LayoutAlign_T, /) -> Row: ...
    @overload
    def bandPosition(self, _: float, /) -> Row: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Row: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Row: ...
    @overload
    def center(self, _: bool, /) -> Row: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Row: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Row: ...
    @overload
    def header(self, _: Header | None, /) -> Row: ...
    @overload
    def header(
        self,
        *,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        labelAngle: Optional[float] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOrient: Optional[SchemaBase | Orient_T] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labels: Optional[bool] = Undefined,
        orient: Optional[SchemaBase | Orient_T] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[SchemaBase | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[SchemaBase | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Row: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | EncodingSortField
        | Sequence[DateTime | Temporal]
        | SortOrder_T
        | None,
        /,
    ) -> Row: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Row: ...
    @overload
    def spacing(self, _: float, /) -> Row: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Row: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Row: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Row: ...
    @overload
    def type(self, _: StandardType_T, /) -> Row: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        align: Optional[SchemaBase | LayoutAlign_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        center: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        header: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | SortOrder_T
            | None
        ] = Undefined,
        spacing: Optional[float] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            align=align,
            bandPosition=bandPosition,
            bin=bin,
            center=center,
            field=field,
            header=header,
            sort=sort,
            spacing=spacing,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class Shape(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefTypeForShapestringnull,
):
    r"""
    Shape schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`TypeForShape`, Literal['nominal', 'ordinal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "shape"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Shape: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Shape: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Shape: ...
    @overload
    def bandPosition(self, _: float, /) -> Shape: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Shape: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Shape: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Shape: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Shape: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> Shape: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Shape: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Shape: ...
    @overload
    def legend(self, _: Legend | None, /) -> Shape: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Shape: ...
    @overload
    def scale(self, _: Scale | None, /) -> Shape: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Shape: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Shape: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Shape: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Shape: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Shape: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Shape: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Shape: ...
    @overload
    def type(self, _: TypeForShape_T, /) -> Shape: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | TypeForShape_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ShapeDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefstringnull
):
    """
    ShapeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "shape"

    @overload
    def bandPosition(self, _: float, /) -> ShapeDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ShapeDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ShapeDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> ShapeDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> ShapeDatum: ...
    @overload
    def type(self, _: Type_T, /) -> ShapeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ShapeValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefTypeForShapestringnull,
):
    """
    ShapeValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, :class:`ConditionalMarkPropFieldOrDatumDefTypeForShape`, :class:`ConditionalParameterMarkPropFieldOrDatumDefTypeForShape`, :class:`ConditionalPredicateMarkPropFieldOrDatumDefTypeForShape`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "shape"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | TypeForShape_T] = Undefined,
    ) -> ShapeValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> ShapeValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | TypeForShape_T] = Undefined,
    ) -> ShapeValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> ShapeValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ShapeValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> ShapeValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> ShapeValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Size(FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber):
    r"""
    Size schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "size"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Size: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Size: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Size: ...
    @overload
    def bandPosition(self, _: float, /) -> Size: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Size: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Size: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Size: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> Size: ...
    @overload
    def condition(self, _: list[core.ConditionalValueDefnumberExprRef], /) -> Size: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Size: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Size: ...
    @overload
    def legend(self, _: Legend | None, /) -> Size: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Size: ...
    @overload
    def scale(self, _: Scale | None, /) -> Size: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Size: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Size: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Size: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Size: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Size: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Size: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Size: ...
    @overload
    def type(self, _: StandardType_T, /) -> Size: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class SizeDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber):
    """
    SizeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "size"

    @overload
    def bandPosition(self, _: float, /) -> SizeDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> SizeDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> SizeDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> SizeDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> SizeDatum: ...
    @overload
    def type(self, _: Type_T, /) -> SizeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class SizeValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    SizeValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "size"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> SizeValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> SizeValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> SizeValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> SizeValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> SizeValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> SizeValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> SizeValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Stroke(
    FieldChannelMixin,
    core.FieldOrDatumDefWithConditionMarkPropFieldDefGradientstringnull,
):
    r"""
    Stroke schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "stroke"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Stroke: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Stroke: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Stroke: ...
    @overload
    def bandPosition(self, _: float, /) -> Stroke: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> Stroke: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Stroke: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Stroke: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> Stroke: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> Stroke: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Stroke: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Stroke: ...
    @overload
    def legend(self, _: Legend | None, /) -> Stroke: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> Stroke: ...
    @overload
    def scale(self, _: Scale | None, /) -> Stroke: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Stroke: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Stroke: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Stroke: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Stroke: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Stroke: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Stroke: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Stroke: ...
    @overload
    def type(self, _: StandardType_T, /) -> Stroke: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefGradientstringnull
):
    """
    StrokeDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "stroke"

    @overload
    def bandPosition(self, _: float, /) -> StrokeDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> StrokeDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> StrokeDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> StrokeDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeDatum: ...
    @overload
    def type(self, _: Type_T, /) -> StrokeDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeValue(
    ValueChannelMixin,
    core.ValueDefWithConditionMarkPropFieldOrDatumDefGradientstringnull,
):
    """
    StrokeValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefGradientstringnullExprRef`, :class:`ConditionalParameterValueDefGradientstringnullExprRef`, :class:`ConditionalPredicateValueDefGradientstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, :class:`Gradient`, :class:`LinearGradient`, :class:`RadialGradient`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "stroke"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> StrokeValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> StrokeValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefGradientstringnullExprRef], /
    ) -> StrokeValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class StrokeDash(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumberArray
):
    r"""
    StrokeDash schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeDash"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> StrokeDash: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> StrokeDash: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> StrokeDash: ...
    @overload
    def bandPosition(self, _: float, /) -> StrokeDash: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> StrokeDash: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberArrayExprRef], /
    ) -> StrokeDash: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> StrokeDash: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def legend(self, _: Legend | None, /) -> StrokeDash: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def scale(self, _: Scale | None, /) -> StrokeDash: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> StrokeDash: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> StrokeDash: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> StrokeDash: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeDash: ...
    @overload
    def type(self, _: StandardType_T, /) -> StrokeDash: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeDashDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumberArray
):
    """
    StrokeDashDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeDash"

    @overload
    def bandPosition(self, _: float, /) -> StrokeDashDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
    ) -> StrokeDashDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
    ) -> StrokeDashDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberArrayExprRef], /
    ) -> StrokeDashDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeDashDatum: ...
    @overload
    def type(self, _: Type_T, /) -> StrokeDashDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeDashValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumberArray
):
    """
    StrokeDashValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberArrayExprRef`, :class:`ConditionalParameterValueDefnumberArrayExprRef`, :class:`ConditionalPredicateValueDefnumberArrayExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, Sequence[float], :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeDash"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeDashValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeDashValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeDashValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeDashValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
    ) -> StrokeDashValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
    ) -> StrokeDashValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberArrayExprRef], /
    ) -> StrokeDashValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class StrokeOpacity(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    StrokeOpacity schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeOpacity"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> StrokeOpacity: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> StrokeOpacity: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> StrokeOpacity: ...
    @overload
    def bandPosition(self, _: float, /) -> StrokeOpacity: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> StrokeOpacity: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> StrokeOpacity: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> StrokeOpacity: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def legend(self, _: Legend | None, /) -> StrokeOpacity: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def scale(self, _: Scale | None, /) -> StrokeOpacity: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> StrokeOpacity: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> StrokeOpacity: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> StrokeOpacity: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeOpacity: ...
    @overload
    def type(self, _: StandardType_T, /) -> StrokeOpacity: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeOpacityDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber
):
    """
    StrokeOpacityDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeOpacity"

    @overload
    def bandPosition(self, _: float, /) -> StrokeOpacityDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacityDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacityDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> StrokeOpacityDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeOpacityDatum: ...
    @overload
    def type(self, _: Type_T, /) -> StrokeOpacityDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeOpacityValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    StrokeOpacityValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeOpacity"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacityValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeOpacityValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> StrokeOpacityValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class StrokeWidth(
    FieldChannelMixin, core.FieldOrDatumDefWithConditionMarkPropFieldDefnumber
):
    r"""
    StrokeWidth schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    legend : dict, :class:`Legend`, None
        An object defining properties of the legend. If ``null``, the legend for the
        encoding channel will be removed.

        **Default value:** If undefined, default `legend properties
        <https://vega.github.io/vega-lite/docs/legend.html>`__ are applied.

        **See also:** `legend <https://vega.github.io/vega-lite/docs/legend.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeWidth"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> StrokeWidth: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> StrokeWidth: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> StrokeWidth: ...
    @overload
    def bandPosition(self, _: float, /) -> StrokeWidth: ...
    @overload
    def bin(self, _: bool | Bin | None, /) -> StrokeWidth: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> StrokeWidth: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> StrokeWidth: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def legend(self, _: Legend | None, /) -> StrokeWidth: ...
    @overload
    def legend(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        clipHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columnPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        columns: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        cornerRadius: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        direction: Optional[SchemaBase | Orientation_T] = Undefined,
        fillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        gradientLength: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gradientStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gradientThickness: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridAlign: Optional[Parameter | SchemaBase | Map | LayoutAlign_T] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        legendY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[SchemaBase | LegendOrient_T] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        rowPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        strokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        symbolDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolFillColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolStrokeColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        symbolStrokeWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        symbolType: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOrient: Optional[Parameter | SchemaBase | Map | Orient_T] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        type: Optional[Literal["symbol", "gradient"]] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def scale(self, _: Scale | None, /) -> StrokeWidth: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> StrokeWidth: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> StrokeWidth: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> StrokeWidth: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeWidth: ...
    @overload
    def type(self, _: StandardType_T, /) -> StrokeWidth: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            legend=legend,
            scale=scale,
            sort=sort,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeWidthDatum(
    DatumChannelMixin, core.FieldOrDatumDefWithConditionDatumDefnumber
):
    """
    StrokeWidthDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeWidth"

    @overload
    def bandPosition(self, _: float, /) -> StrokeWidthDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidthDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidthDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> StrokeWidthDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> StrokeWidthDatum: ...
    @overload
    def type(self, _: Type_T, /) -> StrokeWidthDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class StrokeWidthValue(
    ValueChannelMixin, core.ValueDefWithConditionMarkPropFieldOrDatumDefnumber
):
    """
    StrokeWidthValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, Sequence[dict, :class:`ConditionalValueDefnumberExprRef`, :class:`ConditionalParameterValueDefnumberExprRef`, :class:`ConditionalPredicateValueDefnumberExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : dict, float, :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "strokeWidth"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeWidthValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeWidthValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> StrokeWidthValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> StrokeWidthValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidthValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[float | Parameter | SchemaBase | Map] = Undefined,
    ) -> StrokeWidthValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefnumberExprRef], /
    ) -> StrokeWidthValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Text(FieldChannelMixin, core.FieldOrDatumDefWithConditionStringFieldDefText):
    r"""
    Text schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`, Sequence[dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "text"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Text: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Text: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Text: ...
    @overload
    def bandPosition(self, _: float, /) -> Text: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Text: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Text: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Sequence[str] | Map] = Undefined,
    ) -> Text: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Sequence[str] | Map] = Undefined,
    ) -> Text: ...
    @overload
    def condition(self, _: list[core.ConditionalValueDefTextExprRef], /) -> Text: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Text: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Text: ...
    @overload
    def format(self, _: str, /) -> Text: ...
    @overload
    def format(self, _: Map, /) -> Text: ...
    @overload
    def formatType(self, _: str, /) -> Text: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Text: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Text: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Text: ...
    @overload
    def type(self, _: StandardType_T, /) -> Text: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class TextDatum(DatumChannelMixin, core.FieldOrDatumDefWithConditionStringDatumDefText):
    """
    TextDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    condition : dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`, Sequence[dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "text"

    @overload
    def bandPosition(self, _: float, /) -> TextDatum: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Sequence[str] | Map] = Undefined,
    ) -> TextDatum: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Sequence[str] | Map] = Undefined,
    ) -> TextDatum: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefTextExprRef], /
    ) -> TextDatum: ...
    @overload
    def format(self, _: str, /) -> TextDatum: ...
    @overload
    def format(self, _: Map, /) -> TextDatum: ...
    @overload
    def formatType(self, _: str, /) -> TextDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> TextDatum: ...
    @overload
    def type(self, _: Type_T, /) -> TextDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            condition=condition,
            format=format,
            formatType=formatType,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class TextValue(ValueChannelMixin, core.ValueDefWithConditionStringFieldDefText):
    """
    TextValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalStringFieldDef`, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterStringFieldDef`, :class:`ConditionalPredicateStringFieldDef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`, Sequence[dict, :class:`ConditionalValueDefTextExprRef`, :class:`ConditionalParameterValueDefTextExprRef`, :class:`ConditionalPredicateValueDefTextExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`Text`, Sequence[str], :class:`ExprRef`
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "text"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> TextValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> TextValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Sequence[str] | Map] = Undefined,
    ) -> TextValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Sequence[str] | Map] = Undefined,
    ) -> TextValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefTextExprRef], /
    ) -> TextValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Theta(FieldChannelMixin, core.PositionFieldDefBase):
    r"""
    Theta schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    stack : bool, :class:`StackOffset`, Literal['zero', 'center', 'normalize'], None
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).
        * ``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Theta: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Theta: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Theta: ...
    @overload
    def bandPosition(self, _: float, /) -> Theta: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Theta: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Theta: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Theta: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Theta: ...
    @overload
    def scale(self, _: Scale | None, /) -> Theta: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> Theta: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> Theta: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Theta: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> Theta: ...
    @overload
    def stack(self, _: bool | StackOffset_T | None, /) -> Theta: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Theta: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Theta: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Theta: ...
    @overload
    def type(self, _: StandardType_T, /) -> Theta: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        stack: Optional[bool | SchemaBase | StackOffset_T | None] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            scale=scale,
            sort=sort,
            stack=stack,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ThetaDatum(DatumChannelMixin, core.PositionDatumDefBase):
    """
    ThetaDatum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    stack : bool, :class:`StackOffset`, Literal['zero', 'center', 'normalize'], None
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).
        * ``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta"

    @overload
    def bandPosition(self, _: float, /) -> ThetaDatum: ...
    @overload
    def scale(self, _: Scale | None, /) -> ThetaDatum: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> ThetaDatum: ...
    @overload
    def stack(self, _: bool | StackOffset_T | None, /) -> ThetaDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> ThetaDatum: ...
    @overload
    def type(self, _: Type_T, /) -> ThetaDatum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        stack: Optional[bool | SchemaBase | StackOffset_T | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            bandPosition=bandPosition,
            scale=scale,
            stack=stack,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class ThetaValue(ValueChannelMixin, core.PositionValueDef):
    """
    ThetaValue schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Theta2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    Theta2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Theta2: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Theta2: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Theta2: ...
    @overload
    def bandPosition(self, _: float, /) -> Theta2: ...
    @overload
    def bin(self, _: None, /) -> Theta2: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Theta2: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Theta2: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Theta2: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Theta2: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Theta2: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class Theta2Datum(DatumChannelMixin, core.DatumDef):
    """
    Theta2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta2"

    @overload
    def bandPosition(self, _: float, /) -> Theta2Datum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Theta2Datum: ...
    @overload
    def type(self, _: Type_T, /) -> Theta2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class Theta2Value(ValueChannelMixin, core.PositionValueDef):
    """
    Theta2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "theta2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class Tooltip(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Tooltip schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "tooltip"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Tooltip: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> Tooltip: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> Tooltip: ...
    @overload
    def bandPosition(self, _: float, /) -> Tooltip: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Tooltip: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Tooltip: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Tooltip: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Tooltip: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringExprRef], /
    ) -> Tooltip: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Tooltip: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Tooltip: ...
    @overload
    def format(self, _: str, /) -> Tooltip: ...
    @overload
    def format(self, _: Map, /) -> Tooltip: ...
    @overload
    def formatType(self, _: str, /) -> Tooltip: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Tooltip: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Tooltip: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Tooltip: ...
    @overload
    def type(self, _: StandardType_T, /) -> Tooltip: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class TooltipValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    TooltipValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "tooltip"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> TooltipValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> TooltipValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> TooltipValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> TooltipValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> TooltipValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> TooltipValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> TooltipValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class Url(FieldChannelMixin, core.StringFieldDefWithCondition):
    r"""
    Url schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    condition : dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`, Sequence[dict, :class:`ConditionalValueDefstringExprRef`, :class:`ConditionalParameterValueDefstringExprRef`, :class:`ConditionalPredicateValueDefstringExprRef`]
        One or more value definition(s) with `a parameter or a test predicate
        <https://vega.github.io/vega-lite/docs/condition.html>`__.

        **Note:** A field definition's ``condition`` property can only contain `conditional
        value definitions <https://vega.github.io/vega-lite/docs/condition.html#value>`__
        since Vega-Lite only allows at most one encoded field per encoding channel.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    format : str, dict, :class:`Dict`
        When used with the default ``"number"`` and ``"time"`` format type, the text
        formatting pattern for labels of guides (axes, legends, headers) and text marks.

        * If the format type is ``"number"`` (e.g., for quantitative fields), this is D3's
          `number format pattern <https://github.com/d3/d3-format#locale_format>`__.
        * If the format type is ``"time"`` (e.g., for temporal fields), this is D3's `time
          format pattern <https://github.com/d3/d3-time-format#locale_format>`__.

        See the `format documentation <https://vega.github.io/vega-lite/docs/format.html>`__
        for more examples.

        When used with a `custom formatType
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__, this
        value will be passed as ``format`` alongside ``datum.value`` to the registered
        function.

        **Default value:**  Derived from `numberFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for number
        format and from `timeFormat
        <https://vega.github.io/vega-lite/docs/config.html#format>`__ config for time
        format.
    formatType : str
        The format type for labels. One of ``"number"``, ``"time"``, or a `registered custom
        format type
        <https://vega.github.io/vega-lite/docs/config.html#custom-format-type>`__.

        **Default value:**

        * ``"time"`` for temporal fields and ordinal and nominal fields with ``timeUnit``.
        * ``"number"`` for quantitative fields as well as ordinal and nominal fields without
          ``timeUnit``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "url"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> Url: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> Url: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> Url: ...
    @overload
    def bandPosition(self, _: float, /) -> Url: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> Url: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> Url: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Url: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map] = Undefined,
    ) -> Url: ...
    @overload
    def condition(self, _: list[core.ConditionalValueDefstringExprRef], /) -> Url: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> Url: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> Url: ...
    @overload
    def format(self, _: str, /) -> Url: ...
    @overload
    def format(self, _: Map, /) -> Url: ...
    @overload
    def formatType(self, _: str, /) -> Url: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> Url: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> Url: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> Url: ...
    @overload
    def type(self, _: StandardType_T, /) -> Url: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            condition=condition,
            field=field,
            format=format,
            formatType=formatType,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class UrlValue(ValueChannelMixin, core.StringValueDefWithCondition):
    """
    UrlValue schema wrapper.

    Parameters
    ----------
    condition : dict, :class:`ConditionalMarkPropFieldOrDatumDef`, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterMarkPropFieldOrDatumDef`, :class:`ConditionalPredicateMarkPropFieldOrDatumDef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`, Sequence[dict, :class:`ConditionalValueDefstringnullExprRef`, :class:`ConditionalParameterValueDefstringnullExprRef`, :class:`ConditionalPredicateValueDefstringnullExprRef`]
        A field definition or one or more value definition(s) with a parameter predicate.
    value : str, dict, :class:`ExprRef`, None
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "url"

    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> UrlValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        test: Optional[str | SchemaBase | Map] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> UrlValue: ...
    @overload
    def condition(
        self,
        *,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Map | None] = Undefined,
        empty: Optional[bool] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
    ) -> UrlValue: ...
    @overload
    def condition(
        self,
        *,
        bandPosition: Optional[float] = Undefined,
        datum: Optional[
            Temporal | Parameter | SchemaBase | Map | PrimitiveValue_T
        ] = Undefined,
        empty: Optional[bool] = Undefined,
        legend: Optional[SchemaBase | Map | None] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
    ) -> UrlValue: ...
    @overload
    def condition(
        self,
        *,
        test: Optional[str | SchemaBase | Map] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> UrlValue: ...
    @overload
    def condition(
        self,
        *,
        empty: Optional[bool] = Undefined,
        param: Optional[str | SchemaBase] = Undefined,
        value: Optional[str | Parameter | SchemaBase | Map | None] = Undefined,
    ) -> UrlValue: ...
    @overload
    def condition(
        self, _: list[core.ConditionalValueDefstringnullExprRef], /
    ) -> UrlValue: ...

    def __init__(
        self,
        value,
        condition: Optional[SchemaBase | Sequence[SchemaBase | Map] | Map] = Undefined,
        **kwds,
    ):
        super().__init__(value=value, condition=condition, **kwds)


@with_property_setters
class X(FieldChannelMixin, core.PositionFieldDef):
    r"""
    X schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    axis : dict, :class:`Axis`, None
        An object defining properties of axis's gridlines, ticks and labels. If ``null``,
        the axis for the encoding channel will be removed.

        **Default value:** If undefined, default `axis properties
        <https://vega.github.io/vega-lite/docs/axis.html>`__ are applied.

        **See also:** `axis <https://vega.github.io/vega-lite/docs/axis.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, Literal['binned'], :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    impute : dict, :class:`ImputeParams`, None
        An object defining the properties of the Impute Operation to be applied. The field
        value of the other positional channel is taken as ``key`` of the ``Impute``
        Operation. The field of the ``color`` channel if specified is used as ``groupby`` of
        the ``Impute`` Operation.

        **See also:** `impute <https://vega.github.io/vega-lite/docs/impute.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    stack : bool, :class:`StackOffset`, Literal['zero', 'center', 'normalize'], None
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).
        * ``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`StandardType`, Literal['quantitative', 'ordinal', 'temporal', 'nominal']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> X: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> X: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> X: ...
    @overload
    def axis(self, _: Axis | None, /) -> X: ...
    @overload
    def axis(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        bandPosition: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[bool] = Undefined,
        domainCap: Optional[Parameter | SchemaBase | Map | StrokeCap_T] = Undefined,
        domainColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        domainDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        domainDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        grid: Optional[bool] = Undefined,
        gridCap: Optional[Parameter | SchemaBase | Map | StrokeCap_T] = Undefined,
        gridColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gridDash: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        gridDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelAngle: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelBound: Optional[bool | float | Parameter | SchemaBase | Map] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFlush: Optional[bool | float] = Undefined,
        labelFlushOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labels: Optional[bool] = Undefined,
        maxExtent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        minExtent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[Parameter | SchemaBase | Map | AxisOrient_T] = Undefined,
        position: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        style: Optional[str | Sequence[str]] = Undefined,
        tickBand: Optional[
            Parameter | SchemaBase | Literal["center", "extent"] | Map
        ] = Undefined,
        tickCap: Optional[Parameter | SchemaBase | Map | StrokeCap_T] = Undefined,
        tickColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickDash: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        tickDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickExtra: Optional[bool] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickRound: Optional[bool] = Undefined,
        tickSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        ticks: Optional[bool] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        translate: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> X: ...
    @overload
    def bandPosition(self, _: float, /) -> X: ...
    @overload
    def bin(self, _: bool | Bin | Literal["binned"] | None, /) -> X: ...
    @overload
    def bin(
        self,
        *,
        anchor: Optional[float] = Undefined,
        base: Optional[float] = Undefined,
        binned: Optional[bool] = Undefined,
        divide: Optional[Sequence[float]] = Undefined,
        extent: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        maxbins: Optional[float] = Undefined,
        minstep: Optional[float] = Undefined,
        nice: Optional[bool] = Undefined,
        step: Optional[float] = Undefined,
        steps: Optional[Sequence[float]] = Undefined,
    ) -> X: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> X: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> X: ...
    @overload
    def impute(self, _: Impute | None, /) -> X: ...
    @overload
    def impute(
        self,
        *,
        frame: Optional[Sequence[float | None]] = Undefined,
        keyvals: Optional[SchemaBase | Sequence[Any] | Map] = Undefined,
        method: Optional[SchemaBase | ImputeMethod_T] = Undefined,
        value: Optional[Any] = Undefined,
    ) -> X: ...
    @overload
    def scale(self, _: Scale | None, /) -> X: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> X: ...
    @overload
    def sort(
        self,
        _: Sequence[str]
        | Sequence[bool]
        | Sequence[float]
        | Sequence[DateTime | Temporal]
        | AllSortString_T
        | None,
        /,
    ) -> X: ...
    @overload
    def sort(
        self,
        *,
        field: Optional[str | SchemaBase | Map] = Undefined,
        op: Optional[SchemaBase | NonArgAggregateOp_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> X: ...
    @overload
    def sort(
        self,
        *,
        encoding: Optional[SchemaBase | SortByChannel_T] = Undefined,
        order: Optional[SchemaBase | SortOrder_T | None] = Undefined,
    ) -> X: ...
    @overload
    def stack(self, _: bool | StackOffset_T | None, /) -> X: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> X: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> X: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> X: ...
    @overload
    def type(self, _: StandardType_T, /) -> X: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        axis: Optional[SchemaBase | Map | None] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[bool | SchemaBase | Literal["binned"] | Map | None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        impute: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        sort: Optional[
            SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
            | AllSortString_T
            | None
        ] = Undefined,
        stack: Optional[bool | SchemaBase | StackOffset_T | None] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | StandardType_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            axis=axis,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            impute=impute,
            scale=scale,
            sort=sort,
            stack=stack,
            timeUnit=timeUnit,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class XDatum(DatumChannelMixin, core.PositionDatumDef):
    """
    XDatum schema wrapper.

    Parameters
    ----------
    axis : dict, :class:`Axis`, None
        An object defining properties of axis's gridlines, ticks and labels. If ``null``,
        the axis for the encoding channel will be removed.

        **Default value:** If undefined, default `axis properties
        <https://vega.github.io/vega-lite/docs/axis.html>`__ are applied.

        **See also:** `axis <https://vega.github.io/vega-lite/docs/axis.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    impute : dict, :class:`ImputeParams`, None
        An object defining the properties of the Impute Operation to be applied. The field
        value of the other positional channel is taken as ``key`` of the ``Impute``
        Operation. The field of the ``color`` channel if specified is used as ``groupby`` of
        the ``Impute`` Operation.

        **See also:** `impute <https://vega.github.io/vega-lite/docs/impute.html>`__
        documentation.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    stack : bool, :class:`StackOffset`, Literal['zero', 'center', 'normalize'], None
        Type of stacking offset if the field should be stacked. ``stack`` is only applicable
        for ``x``, ``y``, ``theta``, and ``radius`` channels with continuous domains. For
        example, ``stack`` of ``y`` can be used to customize stacking for a vertical bar
        chart.

        ``stack`` can be one of the following values:

        * ``"zero"`` or ``true``: stacking with baseline offset at zero value of the scale
          (for creating typical stacked `bar
          <https://vega.github.io/vega-lite/docs/stack.html#bar>`__ and `area
          <https://vega.github.io/vega-lite/docs/stack.html#area>`__ chart).
        * ``"normalize"`` - stacking with normalized domain (for creating `normalized
          stacked bar and area charts
          <https://vega.github.io/vega-lite/docs/stack.html#normalized>`__ and pie charts
          `with percentage tooltip
          <https://vega.github.io/vega-lite/docs/arc.html#tooltip>`__).
        * ``"center"`` - stacking with center baseline (for `streamgraph
          <https://vega.github.io/vega-lite/docs/stack.html#streamgraph>`__).
        * ``null`` or ``false`` - No-stacking. This will produce layered `bar
          <https://vega.github.io/vega-lite/docs/stack.html#layered-bar-chart>`__ and area
          chart.

        **Default value:** ``zero`` for plots with all of the following conditions are true:
        (1) the mark is ``bar``, ``area``, or ``arc``; (2) the stacked measure channel (x or
        y) has a linear scale; (3) At least one of non-position channels mapped to an
        unaggregated field that is different from x and y. Otherwise, ``null`` by default.

        **See also:** `stack <https://vega.github.io/vega-lite/docs/stack.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x"

    @overload
    def axis(self, _: Axis | None, /) -> XDatum: ...
    @overload
    def axis(
        self,
        *,
        aria: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        bandPosition: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        description: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[bool] = Undefined,
        domainCap: Optional[Parameter | SchemaBase | Map | StrokeCap_T] = Undefined,
        domainColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        domainDash: Optional[
            Parameter | SchemaBase | Sequence[float] | Map
        ] = Undefined,
        domainDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        format: Optional[str | SchemaBase | Map] = Undefined,
        formatType: Optional[str] = Undefined,
        grid: Optional[bool] = Undefined,
        gridCap: Optional[Parameter | SchemaBase | Map | StrokeCap_T] = Undefined,
        gridColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        gridDash: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        gridDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        gridWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        labelAngle: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        labelBound: Optional[bool | float | Parameter | SchemaBase | Map] = Undefined,
        labelColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        labelExpr: Optional[str] = Undefined,
        labelFlush: Optional[bool | float] = Undefined,
        labelFlushOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        labelFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        labelLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelOverlap: Optional[
            bool | Parameter | SchemaBase | Literal["greedy", "parity"] | Map
        ] = Undefined,
        labelPadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labelSeparation: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        labels: Optional[bool] = Undefined,
        maxExtent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        minExtent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        offset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        orient: Optional[Parameter | SchemaBase | Map | AxisOrient_T] = Undefined,
        position: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        style: Optional[str | Sequence[str]] = Undefined,
        tickBand: Optional[
            Parameter | SchemaBase | Literal["center", "extent"] | Map
        ] = Undefined,
        tickCap: Optional[Parameter | SchemaBase | Map | StrokeCap_T] = Undefined,
        tickColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        tickCount: Optional[
            float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        tickDash: Optional[Parameter | SchemaBase | Sequence[float] | Map] = Undefined,
        tickDashOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickExtra: Optional[bool] = Undefined,
        tickMinStep: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickOffset: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickRound: Optional[bool] = Undefined,
        tickSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        tickWidth: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        ticks: Optional[bool] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        titleAlign: Optional[Parameter | SchemaBase | Map | Align_T] = Undefined,
        titleAnchor: Optional[Parameter | SchemaBase | Map | TitleAnchor_T] = Undefined,
        titleAngle: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleBaseline: Optional[
            Parameter | SchemaBase | Map | TextBaseline_T
        ] = Undefined,
        titleColor: Optional[
            str | Parameter | SchemaBase | Map | ColorName_T | None
        ] = Undefined,
        titleFont: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontSize: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleFontStyle: Optional[str | Parameter | SchemaBase | Map] = Undefined,
        titleFontWeight: Optional[
            Parameter | SchemaBase | Map | FontWeight_T
        ] = Undefined,
        titleLimit: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleLineHeight: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleOpacity: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titlePadding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleX: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        titleY: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        translate: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        values: Optional[
            Parameter
            | SchemaBase
            | Sequence[str]
            | Sequence[bool]
            | Sequence[float]
            | Sequence[Temporal | SchemaBase | Map]
            | Map
        ] = Undefined,
        zindex: Optional[float] = Undefined,
    ) -> XDatum: ...
    @overload
    def bandPosition(self, _: float, /) -> XDatum: ...
    @overload
    def impute(self, _: Impute | None, /) -> XDatum: ...
    @overload
    def impute(
        self,
        *,
        frame: Optional[Sequence[float | None]] = Undefined,
        keyvals: Optional[SchemaBase | Sequence[Any] | Map] = Undefined,
        method: Optional[SchemaBase | ImputeMethod_T] = Undefined,
        value: Optional[Any] = Undefined,
    ) -> XDatum: ...
    @overload
    def scale(self, _: Scale | None, /) -> XDatum: ...
    @overload
    def scale(
        self,
        *,
        align: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        base: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        bins: Optional[SchemaBase | Sequence[float] | Map] = Undefined,
        clamp: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        constant: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domain: Optional[
            Parameter
            | SchemaBase
            | Literal["unaggregated"]
            | Sequence[
                str | bool | float | Temporal | Parameter | SchemaBase | Map | None
            ]
            | Map
        ] = Undefined,
        domainMax: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainMid: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        domainMin: Optional[
            float | Temporal | Parameter | SchemaBase | Map
        ] = Undefined,
        domainRaw: Optional[Parameter | SchemaBase | Map] = Undefined,
        exponent: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        interpolate: Optional[
            Parameter | SchemaBase | Map | ScaleInterpolateEnum_T
        ] = Undefined,
        nice: Optional[
            bool | float | Parameter | SchemaBase | Map | TimeInterval_T
        ] = Undefined,
        padding: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingInner: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        paddingOuter: Optional[float | Parameter | SchemaBase | Map] = Undefined,
        range: Optional[
            SchemaBase
            | Sequence[str | float | Parameter | SchemaBase | Sequence[float] | Map]
            | Map
            | RangeEnum_T
        ] = Undefined,
        rangeMax: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        rangeMin: Optional[str | float | Parameter | SchemaBase | Map] = Undefined,
        reverse: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        round: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
        scheme: Optional[Parameter | SchemaBase | Map | ColorScheme_T] = Undefined,
        type: Optional[SchemaBase | ScaleType_T] = Undefined,
        zero: Optional[bool | Parameter | SchemaBase | Map] = Undefined,
    ) -> XDatum: ...
    @overload
    def stack(self, _: bool | StackOffset_T | None, /) -> XDatum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> XDatum: ...
    @overload
    def type(self, _: Type_T, /) -> XDatum: ...

    def __init__(
        self,
        datum,
        axis: Optional[SchemaBase | Map | None] = Undefined,
        bandPosition: Optional[float] = Undefined,
        impute: Optional[SchemaBase | Map | None] = Undefined,
        scale: Optional[SchemaBase | Map | None] = Undefined,
        stack: Optional[bool | SchemaBase | StackOffset_T | None] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum,
            axis=axis,
            bandPosition=bandPosition,
            impute=impute,
            scale=scale,
            stack=stack,
            title=title,
            type=type,
            **kwds,
        )


@with_property_setters
class XValue(ValueChannelMixin, core.PositionValueDef):
    """
    XValue schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class X2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    X2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> X2: ...
    @overload
    def aggregate(self, *, argmax: Optional[str | SchemaBase] = Undefined) -> X2: ...
    @overload
    def aggregate(self, *, argmin: Optional[str | SchemaBase] = Undefined) -> X2: ...
    @overload
    def bandPosition(self, _: float, /) -> X2: ...
    @overload
    def bin(self, _: None, /) -> X2: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> X2: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> X2: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> X2: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> X2: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> X2: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class X2Datum(DatumChannelMixin, core.DatumDef):
    """
    X2Datum schema wrapper.

    Parameters
    ----------
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    datum : str, bool, dict, float, :class:`ExprRef`, :class:`DateTime`, :class:`RepeatRef`, :class:`PrimitiveValue`, None
        A constant value in data domain.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    type : :class:`Type`, Literal['quantitative', 'ordinal', 'temporal', 'nominal', 'geojson']
        The type of measurement (``"quantitative"``, ``"temporal"``, ``"ordinal"``, or
        ``"nominal"``) for the encoded field or constant value (``datum``). It can also be a
        ``"geojson"`` type for encoding `'geoshape'
        <https://vega.github.io/vega-lite/docs/geoshape.html>`__.

        Vega-Lite automatically infers data types in many cases as discussed below. However,
        type is required for a field if: (1) the field is not nominal and the field encoding
        has no specified ``aggregate`` (except ``argmin`` and ``argmax``), ``bin``, scale
        type, custom ``sort`` order, nor ``timeUnit`` or (2) if you wish to use an ordinal
        scale for a field with ``bin`` or ``timeUnit``.

        **Default value:**

        1) For a data ``field``, ``"nominal"`` is the default data type unless the field
        encoding has ``aggregate``, ``channel``, ``bin``, scale type, ``sort``, or
        ``timeUnit`` that satisfies the following criteria:

        * ``"quantitative"`` is the default type if (1) the encoded field contains ``bin``
          or ``aggregate`` except ``"argmin"`` and ``"argmax"``, (2) the encoding channel is
          ``latitude`` or ``longitude`` channel or (3) if the specified scale type is `a
          quantitative scale <https://vega.github.io/vega-lite/docs/scale.html#type>`__.
        * ``"temporal"`` is the default type if (1) the encoded field contains ``timeUnit``
          or (2) the specified scale type is a time or utc scale
        * ``"ordinal"`` is the default type if (1) the encoded field contains a `custom sort
          order
          <https://vega.github.io/vega-lite/docs/sort.html#specifying-custom-sort-order>`__,
          (2) the specified scale type is an ordinal/point/band scale, or (3) the encoding
          channel is ``order``.

        2) For a constant value in data domain (``datum``):

        * ``"quantitative"`` if the datum is a number
        * ``"nominal"`` if the datum is a string
        * ``"temporal"`` if the datum is `a date time object
          <https://vega.github.io/vega-lite/docs/datetime.html>`__

        **Note:**

        * Data ``type`` describes the semantics of the data rather than the primitive data
          types (number, string, etc.). The same primitive data type can have different
          types of measurement. For example, numeric data can represent quantitative,
          ordinal, or nominal data.
        * Data values for a temporal field can be either a date-time string (e.g.,
          ``"2015-03-07 12:32:17"``, ``"17:01"``, ``"2015-03-16"``. ``"2015"``) or a
          timestamp number (e.g., ``1552199579097``).
        * When using with `bin <https://vega.github.io/vega-lite/docs/bin.html>`__, the
          ``type`` property can be either ``"quantitative"`` (for using a linear bin scale)
          or `"ordinal" (for using an ordinal bin scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `timeUnit
          <https://vega.github.io/vega-lite/docs/timeunit.html>`__, the ``type`` property
          can be either ``"temporal"`` (default, for using a temporal scale) or `"ordinal"
          (for using an ordinal scale)
          <https://vega.github.io/vega-lite/docs/type.html#cast-bin>`__.
        * When using with `aggregate
          <https://vega.github.io/vega-lite/docs/aggregate.html>`__, the ``type`` property
          refers to the post-aggregation data type. For example, we can calculate count
          ``distinct`` of a categorical field ``"cat"`` using ``{"aggregate": "distinct",
          "field": "cat"}``. The ``"type"`` of the aggregate output is ``"quantitative"``.
        * Secondary channels (e.g., ``x2``, ``y2``, ``xError``, ``yError``) do not have
          ``type`` as they must have exactly the same type as their primary channels (e.g.,
          ``x``, ``y``).

        **See also:** `type <https://vega.github.io/vega-lite/docs/type.html>`__
        documentation.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x2"

    @overload
    def bandPosition(self, _: float, /) -> X2Datum: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> X2Datum: ...
    @overload
    def type(self, _: Type_T, /) -> X2Datum: ...

    def __init__(
        self,
        datum,
        bandPosition: Optional[float] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        type: Optional[SchemaBase | Type_T] = Undefined,
        **kwds,
    ):
        super().__init__(
            datum=datum, bandPosition=bandPosition, title=title, type=type, **kwds
        )


@with_property_setters
class X2Value(ValueChannelMixin, core.PositionValueDef):
    """
    X2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : dict, float, :class:`ExprRef`, Literal['height', 'width']
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "x2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class XError(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    XError schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "xError"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> XError: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> XError: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> XError: ...
    @overload
    def bandPosition(self, _: float, /) -> XError: ...
    @overload
    def bin(self, _: None, /) -> XError: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> XError: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> XError: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> XError: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> XError: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> XError: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class XErrorValue(ValueChannelMixin, core.ValueDefnumber):
    """
    XErrorValue schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : float
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "xError"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class XError2(FieldChannelMixin, core.SecondaryFieldDef):
    r"""
    XError2 schema wrapper.

    A field definition of a secondary channel that shares a scale with another primary channel.
    For example, ``x2``, ``xError`` and ``xError2`` share the same scale with ``x``.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **Notes**:

        1) You can customize the default field title format by providing the `fieldTitle
        <https://vega.github.io/vega-lite/docs/config.html#top-level-config>`__ property in
        the `config <https://vega.github.io/vega-lite/docs/config.html>`__ or `fieldTitle
        function via the compile function's options
        <https://vega.github.io/vega-lite/usage/compile.html#field-title>`__.

        2) If both field definition's ``title`` and axis, header, or legend ``title`` are
        defined, axis/header/legend title will be used.
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "xError2"

    @overload
    def aggregate(self, _: NonArgAggregateOp_T, /) -> XError2: ...
    @overload
    def aggregate(
        self, *, argmax: Optional[str | SchemaBase] = Undefined
    ) -> XError2: ...
    @overload
    def aggregate(
        self, *, argmin: Optional[str | SchemaBase] = Undefined
    ) -> XError2: ...
    @overload
    def bandPosition(self, _: float, /) -> XError2: ...
    @overload
    def bin(self, _: None, /) -> XError2: ...
    @overload
    def field(self, _: str | RepeatRef, /) -> XError2: ...
    @overload
    def field(
        self,
        *,
        repeat: Optional[Literal["row", "column", "repeat", "layer"]] = Undefined,
    ) -> XError2: ...
    @overload
    def timeUnit(
        self,
        _: TimeUnitParams | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T,
        /,
    ) -> XError2: ...
    @overload
    def timeUnit(
        self,
        *,
        binned: Optional[bool] = Undefined,
        maxbins: Optional[float] = Undefined,
        step: Optional[float] = Undefined,
        unit: Optional[SchemaBase | MultiTimeUnit_T | SingleTimeUnit_T] = Undefined,
        utc: Optional[bool] = Undefined,
    ) -> XError2: ...
    @overload
    def title(self, _: str | Sequence[str] | None, /) -> XError2: ...

    def __init__(
        self,
        shorthand: Optional[str | SchemaBase | Sequence[str] | Map] = Undefined,
        aggregate: Optional[SchemaBase | Map | NonArgAggregateOp_T] = Undefined,
        bandPosition: Optional[float] = Undefined,
        bin: Optional[None] = Undefined,
        field: Optional[str | SchemaBase | Map] = Undefined,
        timeUnit: Optional[
            SchemaBase | Map | MultiTimeUnit_T | BinnedTimeUnit_T | SingleTimeUnit_T
        ] = Undefined,
        title: Optional[str | SchemaBase | Sequence[str] | None] = Undefined,
        **kwds,
    ):
        super().__init__(
            shorthand=shorthand,
            aggregate=aggregate,
            bandPosition=bandPosition,
            bin=bin,
            field=field,
            timeUnit=timeUnit,
            title=title,
            **kwds,
        )


@with_property_setters
class XError2Value(ValueChannelMixin, core.ValueDefnumber):
    """
    XError2Value schema wrapper.

    Definition object for a constant value (primitive value or gradient definition) of an
    encoding channel.

    Parameters
    ----------
    value : float
        A constant value in visual domain (e.g., ``"red"`` / ``"#0099ff"`` / `gradient
        definition <https://vega.github.io/vega-lite/docs/types.html#gradient>`__ for color,
        values between ``0`` to ``1`` for opacity).
    """

    _class_is_valid_at_instantiation = False
    _encoding_name = "xError2"

    def __init__(self, value, **kwds):
        super().__init__(value=value, **kwds)


@with_property_setters
class XOffset(FieldChannelMixin, core.ScaleFieldDef):
    r"""
    XOffset schema wrapper.

    Parameters
    ----------
    shorthand : str, dict, Sequence[str], :class:`RepeatRef`
        shorthand for field, aggregate, and type
    aggregate : dict, :class:`Aggregate`, :class:`ArgmaxDef`, :class:`ArgminDef`, :class:`NonArgAggregateOp`, Literal['average', 'count', 'distinct', 'max', 'mean', 'median', 'min', 'missing', 'product', 'q1', 'q3', 'ci0', 'ci1', 'stderr', 'stdev', 'stdevp', 'sum', 'valid', 'values', 'variance', 'variancep', 'exponential', 'exponentialb']
        Aggregation function for the field (e.g., ``"mean"``, ``"sum"``, ``"median"``,
        ``"min"``, ``"max"``, ``"count"``).

        **Default value:** ``undefined`` (None)

        **See also:** `aggregate <https://vega.github.io/vega-lite/docs/aggregate.html>`__
        documentation.
    bandPosition : float
        Relative position on a band of a stacked, binned, time unit, or band scale. For
        example, the marks will be positioned at the beginning of the band if set to ``0``,
        and at the middle of the band if set to ``0.5``.
    bin : bool, dict, :class:`BinParams`, None
        A flag for binning a ``quantitative`` field, `an object defining binning parameters
        <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__, or indicating
        that the data for ``x`` or ``y`` channel are binned before they are imported into
        Vega-Lite (``"binned"``).

        * If ``true``, default `binning parameters
          <https://vega.github.io/vega-lite/docs/bin.html#bin-parameters>`__ will be
          applied.

        * If ``"binned"``, this indicates that the data for the ``x`` (or ``y``) channel are
          already binned. You can map the bin-start field to ``x`` (or ``y``) and the
          bin-end field to ``x2`` (or ``y2``). The scale and axis will be formatted similar
          to binning in Vega-Lite.  To adjust the axis ticks based on the bin step, you can
          also set the axis's `tickMinStep
          <https://vega.github.io/vega-lite/docs/axis.html#ticks>`__ property.

        **Default value:** ``false``

        **See also:** `bin <https://vega.github.io/vega-lite/docs/bin.html>`__
        documentation.
    field : str, dict, :class:`Field`, :class:`FieldName`, :class:`RepeatRef`
        **Required.** A string defining the name of the field from which to pull a data
        value or an object defining iterated values from the `repeat
        <https://vega.github.io/vega-lite/docs/repeat.html>`__ operator.

        **See also:** `field <https://vega.github.io/vega-lite/docs/field.html>`__
        documentation.

        **Notes:** 1)  Dots (``.``) and brackets (``[`` and ``]``) can be used to access
        nested objects (e.g., ``"field": "foo.bar"`` and ``"field": "foo['bar']"``). If
        field names contain dots or brackets but are not nested, you can use ``\\`` to
        escape dots and brackets (e.g., ``"a\\.b"`` and ``"a\\[0\\]"``). See more details
        about escaping in the `field documentation
        <https://vega.github.io/vega-lite/docs/field.html>`__. 2) ``field`` is not required
        if ``aggregate`` is ``count``.
    scale : dict, :class:`Scale`, None
        An object defining properties of the channel's scale, which is the function that
        transforms values in the data domain (numbers, dates, strings, etc) to visual values
        (pixels, colors, sizes) of the encoding channels.

        If ``null``, the scale will be `disabled and the data value will be directly encoded
        <https://vega.github.io/vega-lite/docs/scale.html#disable>`__.

        **Default value:** If undefined, default `scale properties
        <https://vega.github.io/vega-lite/docs/scale.html>`__ are applied.

        **See also:** `scale <https://vega.github.io/vega-lite/docs/scale.html>`__
        documentation.
    sort : dict, :class:`Sort`, Sequence[str], Sequence[bool], Sequence[float], :class:`SortArray`, :class:`SortOrder`, :class:`AllSortString`, :class:`SortByChannel`, :class:`SortByEncoding`, :class:`EncodingSortField`, :class:`SortByChannelDesc`, Sequence[dict, :class:`DateTime`], Literal['-x', '-y', '-color', '-fill', '-stroke', '-strokeWidth', '-size', '-shape', '-fillOpacity', '-strokeOpacity', '-opacity', '-text', 'ascending', 'descending', 'x', 'y', 'color', 'fill', 'stroke', 'strokeWidth', 'size', 'shape', 'fillOpacity', 'strokeOpacity', 'opacity', 'text'], None
        Sort order for the encoded field.

        For continuous fields (quantitative or temporal), ``sort`` can be either
        ``"ascending"`` or ``"descending"``.

        For discrete fields, ``sort`` can be one of the following:

        * ``"ascending"`` or ``"descending"`` -- for sorting by the values' natural order in
          JavaScript.
        * `A string indicating an encoding channel name to sort by
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__ (e.g.,
          ``"x"`` or ``"y"``) with an optional minus prefix for descending sort (e.g.,
          ``"-x"`` to sort by x-field, descending). This channel string is short-form of `a
          sort-by-encoding definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-by-encoding>`__. For
          example, ``"sort": "-x"`` is equivalent to ``"sort": {"encoding": "x", "order":
          "descending"}``.
        * `A sort field definition
          <https://vega.github.io/vega-lite/docs/sort.html#sort-field>`__ for sorting by
          another field.
        * `An array specifying the field values in preferred order
          <https://vega.github.io/vega-lite/docs/sort.html#sort-array>`__. In this case, the
          sort order will obey the values in the array, followed by any unspecified values
          in their original order. For discrete time field, values in the sort array can be
          `date-time definition objects
          <https://vega.github.io/vega-lite/docs/datetime.html>`__. In addition, for time
          units ``"month"`` and ``"day"``, the values can be the month or day names (case
          insensitive) or their 3-letter initials (e.g., ``"Mon"``, ``"Tue"``).
        * ``null`` indicating no sort.

        **Default value:** ``"ascending"``

        **Note:** ``null`` and sorting by another channel is not supported for ``row`` and
        ``column``.

        **See also:** `sort <https://vega.github.io/vega-lite/docs/sort.html>`__
        documentation.
    timeUnit : dict, :class:`TimeUnit`, :class:`MultiTimeUnit`, :class:`BinnedTimeUnit`, :class:`SingleTimeUnit`, :class:`TimeUnitParams`, :class:`UtcMultiTimeUnit`, :class:`UtcSingleTimeUnit`, :class:`LocalMultiTimeUnit`, :class:`LocalSingleTimeUnit`, Literal['binnedutcyear', 'binnedutcyearquarter', 'binnedutcyearquartermonth', 'binnedutcyearmonth', 'binnedutcyearmonthdate', 'binnedutcyearmonthdatehours', 'binnedutcyearmonthdatehoursminutes', 'binnedutcyearmonthdatehoursminutesseconds', 'binnedutcyearweek', 'binnedutcyearweekday', 'binnedutcyearweekdayhours', 'binnedutcyearweekdayhoursminutes', 'binnedutcyearweekdayhoursminutesseconds', 'binnedutcyeardayofyear', 'binnedyear', 'binnedyearquarter', 'binnedyearquartermonth', 'binnedyearmonth', 'binnedyearmonthdate', 'binnedyearmonthdatehours', 'binnedyearmonthdatehoursminutes', 'binnedyearmonthdatehoursminutesseconds', 'binnedyearweek', 'binnedyearweekday', 'binnedyearweekdayhours', 'binnedyearweekdayhoursminutes', 'binnedyearweekdayhoursminutesseconds', 'binnedyeardayofyear', 'utcyear', 'utcquarter', 'utcmonth', 'utcweek', 'utcday', 'utcdayofyear', 'utcdate', 'utchours', 'utcminutes', 'utcseconds', 'utcmilliseconds', 'year', 'quarter', 'month', 'week', 'day', 'dayofyear', 'date', 'hours', 'minutes', 'seconds', 'milliseconds', 'utcyearquarter', 'utcyearquartermonth', 'utcyearmonth', 'utcyearmonthdate', 'utcyearmonthdatehours', 'utcyearmonthdatehoursminutes', 'utcyearmonthdatehoursminutesseconds', 'utcyearweek', 'utcyearweekday', 'utcyearweekdayhours', 'utcyearweekdayhoursminutes', 'utcyearweekdayhoursminutesseconds', 'utcyeardayofyear', 'utcquartermonth', 'utcmonthdate', 'utcmonthdatehours', 'utcmonthdatehoursminutes', 'utcmonthdatehoursminutesseconds', 'utcweekday', 'utcweekdayhours', 'utcweekdayhoursminutes', 'utcweekdayhoursminutesseconds', 'utcdayhours', 'utcdayhoursminutes', 'utcdayhoursminutesseconds', 'utchoursminutes', 'utchoursminutesseconds', 'utcminutesseconds', 'utcsecondsmilliseconds', 'yearquarter', 'yearquartermonth', 'yearmonth', 'yearmonthdate', 'yearmonthdatehours', 'yearmonthdatehoursminutes', 'yearmonthdatehoursminutesseconds', 'yearweek', 'yearweekday', 'yearweekdayhours', 'yearweekdayhoursminutes', 'yearweekdayhoursminutesseconds', 'yeardayofyear', 'quartermonth', 'monthdate', 'monthdatehours', 'monthdatehoursminutes', 'monthdatehoursminutesseconds', 'weekday', 'weekdayhours', 'weekdayhoursminutes', 'weekdayhoursminutesseconds', 'dayhours', 'dayhoursminutes', 'dayhoursminutesseconds', 'hoursminutes', 'hoursminutesseconds', 'minutesseconds', 'secondsmilliseconds']
        Time unit (e.g., ``year``, ``yearmonth``, ``month``, ``hours``) for a temporal
        field. or `a temporal field that gets casted as ordinal
        <https://vega.github.io/vega-lite/docs/type.html#cast>`__.

        **Default value:** ``undefined`` (None)

        **See also:** `timeUnit <https://vega.github.io/vega-lite/docs/timeunit.html>`__
        documentation.
    title : str, :class:`Text`, Sequence[str], None
        A title for the field. If ``null``, the title will be removed.

        **Default value:**  derived from the field's name and transformation function
        (``aggregate``, ``bin`` and ``timeUnit``). If the field has an aggregate function,
        the function is displayed as part of the title (e.g., ``"Sum of Profit"``). If the
        field is binned or has a time unit applied, the applied function is shown in
        parentheses (e.g., ``"Profit (binned)"``, ``"Transaction Date (year-month)"``).
        Otherwise, the title is simply the field name.

        **