# (c) Copyright 2009-2015. CodeWeavers, Inc.

import threading

import Foundation # pylint: disable=E0401
import objc # pylint: disable=E0401

import cxurlget_base

_NSURLSessionDelegate = objc.protocolNamed('NSURLSessionDelegate')
_NSURLSessionTaskDelegate = objc.protocolNamed('NSURLSessionTaskDelegate')
_NSURLSessionDataDelegate = objc.protocolNamed('NSURLSessionDataDelegate')

NSURLSessionResponseCancel = Foundation.NSURLSessionResponseCancel
NSURLSessionResponseAllow = Foundation.NSURLSessionResponseAllow
NSURLSessionResponseBecomeDownload = Foundation.NSURLSessionResponseBecomeDownload
NSURLSessionResponseBecomeStream = Foundation.NSURLSessionResponseBecomeStream

class _UrlGetterDelegate(Foundation.NSObject):
    __pyobjc_protocols__ = [_NSURLSessionDelegate, _NSURLSessionTaskDelegate, _NSURLSessionDataDelegate]

    def initWithOwner_(self, owner):
        self = objc.super(_UrlGetterDelegate, self).init()
        if self is None:
            return None

        self.owner = owner
        return self

    def URLSession_didBecomeInvalidWithError_(self, _session, _error):
        self.owner.event.set()

    def URLSession_task_didCompleteWithError_(self, _session, _task, error):
        try:
            if error:
                self.owner.notify_failed(self.owner, cxurlget_base.UrlError(error.localizedDescription()))
            else:
                self.owner.finished = True
                self.owner.notify_finished(self.owner)
        except Exception as exception:
            self.owner.exception = exception
        self.owner.event.set()

    def URLSession_dataTask_didReceiveResponse_completionHandler_(self, _session, _dataTask, response, completionHandler):
        if isinstance(response, Foundation.NSHTTPURLResponse):
            code = response.statusCode()
            if 300 <= code <= 399:
                # Don't try to populate information from headers from a redirect
                completionHandler(NSURLSessionResponseAllow)
                return
            if not 200 <= code <= 299:
                self.owner.notify_failed(self.owner, cxurlget_base.HttpError(
                    code, Foundation.NSHTTPURLResponse.localizedStringForStatusCode_(code)))
                self.owner.event.set()
                completionHandler(NSURLSessionResponseCancel)
                return

            self.owner.last_modified = response.allHeaderFields().objectForKey_('Last-Modified')
            self.owner.etag = response.allHeaderFields().objectForKey_('ETag')

        length = response.expectedContentLength()
        if length != Foundation.NSURLResponseUnknownLength:
            self.owner.bytes_total = length

        basename = str(response.suggestedFilename())
        if basename != 'unknown':
            self.owner.basename = basename

        try:
            self.owner.notify_progress(self.owner)
        except Exception as e:
            if not isinstance(e, cxurlget_base.StopDownload):
                self.owner.notify_failed(self.owner, e)
            self.owner.event.set()
            completionHandler(NSURLSessionResponseCancel)
            return

        completionHandler(NSURLSessionResponseAllow)

    def URLSession_dataTask_didReceiveData_(self, _session, dataTask, data):
        try:
            self.owner.outfile.write(data)
            self.owner.bytes_downloaded = self.owner.bytes_downloaded + len(data)
            self.owner.notify_progress(self.owner)
        except Exception as e:
            if not isinstance(e, cxurlget_base.StopDownload):
                self.owner.notify_failed(self.owner, e)
            dataTask.cancel()
            self.owner.event.set()

class UrlGetter(cxurlget_base.UrlGetterBase):
    @classmethod
    def make_cookie_storage(cls):
        # Freshly alloc/init'd instances of NSHTTPCookieStorage are not
        # functional, and sharedHTTPCookieStorage is exactly what we're
        # trying to avoid here. So we pull the in-memory cookie storage off
        # an ephemeral configuration.
        config = Foundation.NSURLSessionConfiguration.ephemeralSessionConfiguration()
        return config.HTTPCookieStorage()

    def __init__(self, *args, **kwargs):
        cxurlget_base.UrlGetterBase.__init__(self, *args, **kwargs)

    def fetch(self):
        self.event = threading.Event()
        self.exception = None
        config = Foundation.NSURLSessionConfiguration.ephemeralSessionConfiguration()
        if self.timeout is not None:
            config.setTimeoutIntervalForResource_(self.timeout)
        if self.cookie_storage:
            config.setHTTPCookieStorage_(self.cookie_storage)
        config.setHTTPAdditionalHeaders_({'User-Agent': self.user_agent, 'Accept': '*/*'})
        delegate = _UrlGetterDelegate.alloc().initWithOwner_(self) # pylint: disable=E1101
        session = Foundation.NSURLSession.sessionWithConfiguration_delegate_delegateQueue_(config, delegate, None)
        try:
            url = Foundation.NSURL.URLWithString_(self.url)

            request = Foundation.NSMutableURLRequest.alloc().initWithURL_(url)
            request.setHTTPMethod_(self.verb)
            if self.data is not None:
                request.setHTTPBody_(self.data)

            task = session.dataTaskWithRequest_(request)

            task.resume()

            self.event.wait()
        finally:
            session.invalidateAndCancel()
            self.outfile.close()
            self.outfile = None

        if self.exception is not None:
            raise self.exception # pylint: disable=E0702
