!{\src2tex{textfont=tt}}
!!****f* ABINIT/wvl_vtorho
!! NAME
!! wvl_vtorho
!!
!! FUNCTION
!! Heart of the wavelet resolution, compute new wavefunctions mixed witf previous
!! by computing the gradient of the wavefunctions knowing the external potential.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2007 ABINIT group (DC)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  dtset <type(dataset_type)>=input variables.
!!  istep=id of the current iteration (first is 1).
!!  mpi_enreg=informations about MPI parallelization
!!  occ(dtset%mband)=occupation numbers.
!!  proj <type(wvl_projector_type)>=projectors informations for wavelets.
!!  psps <type(pseudopotential_type)>=variables related to pseudopotentials
!!  vtrial(dtset%nfft)=external potential.
!!
!! OUTPUT
!!
!! SIDE EFFECTS
!!  energies <type(energies_type)>=storage for energies computed here :
!!   | e_kinetic(OUT)=kinetic energy part of total energy
!!   | e_localpsp(OUT)=local pseudopotential part of total energy
!!   | e_nonlocalpsp(OUT)=nonlocal pseudopotential part of total energy
!!  residm=max value for gradient in the minimisation process.
!!  rhor(dtset%nfft)=electron density in r space
!!  wfs <type(wvl_projector_type)>=wavefunctions informations for wavelets.
!!  xred(3,natom)=reduced dimensionless atomic coordinates (in fact IN but here
!!                because of INOUT xredxcart() behavior).
!!
!! PARENTS
!!      vtorho
!!
!! CHILDREN
!!      applylocpotkinall,applyprojectorsall,daxpy,diisstp,leave_new
!!      mpi_allreduce,orthoconstraint,orthoconstraint_p,orthon,orthon_p
!!      preconditionall,solveks,sumrho,transallwaves,untransallwaves,wrtout
!!
!! SOURCE
#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine wvl_vtorho(dtset, energies, grnl, istep, mpi_enreg, occ, proj, &
     & psps, residm, rhor, rprimd, vtrial, wfs, xred)

  use defs_basis
  use defs_datatypes
#if defined HAVE_BIGDFT
  use libbigdft
#endif

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_12geometry
#endif
!End of the abilint section

  implicit none

!Arguments -------------------------------
  type(dataset_type), intent(inout)         :: dtset
  type(energies_type), intent(inout)        :: energies
  integer, intent(in)                       :: istep
  type(MPI_type), intent(in)                :: mpi_enreg
  type(wvl_projectors_type), intent(in)     :: proj
  type(pseudopotential_type), intent(in)    :: psps
  real(dp), intent(inout)                     :: residm
  type(wvl_wf_type), intent(inout)          :: wfs
  real(dp), intent(in)                      :: occ(dtset%mband)
  real(dp), intent(inout)                   :: rhor(dtset%nfft)
  real(dp), intent(in)                      :: rprimd(3, 3)
  real(dp), intent(in)                      :: vtrial(dtset%nfft)
  real(dp), intent(inout)                     :: grnl(3, dtset%natom)
  real(dp), intent(inout)                   :: xred(3, dtset%natom)

!Local variables-------------------------------
  real(dp), save        :: alpha, etotal_local
  character(len = 500)  :: message
  real(dp)              :: epot_sum, ekin_sum, eproj_sum, energy
  real(dp)              :: epot_sum_p, ekin_sum_p, eproj_sum_p
  real(dp)              :: scprsum, gnrm
  real(dp)              :: scprsum_p, gnrm_p
  real(dp)              :: scpr, dnrm2
  integer               :: iorb, mids, k, istepDIIS, ia, igeo
  logical               :: parallel
  real(dp), allocatable :: hpsi(:, :), xcart(:, :), gxyz(:, :)
#if defined MPI
  integer               :: ierr
  real(dp)              :: wrkallred(5,2) 
  real(dp), allocatable :: hpsit(:, :)

  include "mpif.h"
#endif
  
#if defined MPI
    parallel = .true.
#else
    parallel = .false.
#endif
  
#if defined HAVE_BIGDFT
  ! Initialisation of mixing parameter alpha
  if (istep == 1) then
    alpha        = real(1., dp)
    etotal_local = real(1.d100, dp)
  end if  
  
  ! Set the current index for DIIS storage.
  if (dtset%irdwfk /= 0 .or. dtset%getwfk /= 0) then
    istepDIIS = istep
  else
    istepDIIS = istep - 1
  end if
  if (dtset%nwfshist > 0) mids = modulo(istepDIIS - 1, dtset%nwfshist) + 1

  ! In the case of first run and wavefunctions computed from atomic orbitals,
  ! the system is diagonalised from KS hamiltionian. wfs%psi_init is used instead
  ! of wfs%psi...
  if (wfs%mband_init > 0) then
    allocate(hpsi(wfs%keys%nvctr(0) + 7 * wfs%keys%nvctr(1), wfs%mbandp_init))
    write(message,*) ' First step, compute eigenvectors from atomic orbitals.'
    call wrtout(06, message, 'COLL')
  end if

  ! local potential and kinetic energy for all orbitals
  ! BigDFT method.
  if (wfs%mband_init > 0) then
    call applylocpotkinall(mpi_enreg%me, wfs%mband_init, wfs%mbandp_init, &
                         & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
                         & dtset%wvl_internal%nSize(3), dtset%wvl_internal%fineGrid(1, 1), &
                         & dtset%wvl_internal%fineGrid(2, 1), dtset%wvl_internal%fineGrid(1, 2), &
                         & dtset%wvl_internal%fineGrid(2, 2), dtset%wvl_internal%fineGrid(1, 3), &
                         & dtset%wvl_internal%fineGrid(2, 3), dtset%wvl_hgrid, wfs%occ_init, &
                         & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%nvctr(0), &
                         & wfs%keys%nvctr(1), wfs%keys%keyg, wfs%keys%keyv, &
                         & wfs%ibyz_c, wfs%ibxz_c, wfs%ibxy_c, &
                         & wfs%ibyz_f, wfs%ibxz_f, wfs%ibxy_f, wfs%psi_init, &
                         & vtrial, hpsi, epot_sum_p, ekin_sum_p)
  else
    call applylocpotkinall(mpi_enreg%me, dtset%mband, wfs%mbandp, &
                         & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
                         & dtset%wvl_internal%nSize(3), dtset%wvl_internal%fineGrid(1, 1), &
                         & dtset%wvl_internal%fineGrid(2, 1), dtset%wvl_internal%fineGrid(1, 2), &
                         & dtset%wvl_internal%fineGrid(2, 2), dtset%wvl_internal%fineGrid(1, 3), &
                         & dtset%wvl_internal%fineGrid(2, 3), dtset%wvl_hgrid, occ, &
                         & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%nvctr(0), &
                         & wfs%keys%nvctr(1), wfs%keys%keyg, wfs%keys%keyv, &
                         & wfs%ibyz_c, wfs%ibxz_c, wfs%ibxy_c, &
                         & wfs%ibyz_f, wfs%ibxz_f, wfs%ibxy_f, wfs%psi, &
                         & vtrial, wfs%hpsi, epot_sum_p, ekin_sum_p)
  end if

  ! apply all PSP projectors for all orbitals belonging to iproc
  if (wfs%mband_init > 0) then
    call applyprojectorsall(mpi_enreg%me, dtset%ntypat, dtset%natom, &
                          & dtset%typat, psps%gth_params%psppar, psps%pspcod, &
                          & wfs%occ_init, proj%nprojel, proj%nproj, &
                          & proj%keys%nseg, proj%keys%keyg, proj%keys%keyv, &
                          & proj%keys%nvctr, proj%proj, wfs%mband_init, wfs%mbandp_init, &
                          & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%keyg, &
                          & wfs%keys%keyv, wfs%keys%nvctr(0), wfs%keys%nvctr(1), &
                          & wfs%psi_init, hpsi, eproj_sum_p)
  else
    call applyprojectorsall(mpi_enreg%me, dtset%ntypat, dtset%natom, &
                          & dtset%typat, psps%gth_params%psppar, psps%pspcod, &
                          & occ, proj%nprojel, proj%nproj, &
                          & proj%keys%nseg, proj%keys%keyg, proj%keys%keyv, &
                          & proj%keys%nvctr, proj%proj, dtset%mband, wfs%mbandp, &
                          & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%keyg, &
                          & wfs%keys%keyv, wfs%keys%nvctr(0), wfs%keys%nvctr(1), &
                          & wfs%psi, wfs%hpsi, eproj_sum_p)
  end if
                     
  ! In the case of first run from scratch, the system is diagonalised from KS hamiltionian.
  if (wfs%mband_init > 0) then
    ! norm of gradient
    gnrm_p = 0.d0
    do iorb = mpi_enreg%me * wfs%mbandp_init + 1, &
         & min((mpi_enreg%me + 1) * wfs%mbandp_init, wfs%mband_init), 1
       scpr = DNRM2(wfs%keys%nvctr(0) + 7 * wfs%keys%nvctr(1), &
            & hpsi(1, iorb - mpi_enreg%me * wfs%mbandp_init), 1) 
       gnrm_p = gnrm_p + scpr ** 2
    enddo

#if defined MPI
    wrkallred(1, 2) = epot_sum_p
    wrkallred(2, 2) = ekin_sum_p
    wrkallred(3, 2) = eproj_sum_p
    wrkallred(4, 2) = gnrm_p
    call MPI_ALLREDUCE(wrkallred(1,2), wrkallred(1,1), 4, &
                     & MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
    energies%e_localpsp    = wrkallred(1, 1)
    energies%e_kinetic     = wrkallred(2, 1)
    energies%e_nonlocalpsp = wrkallred(3, 1)
    gnrm                   = wrkallred(4, 1)
#else
    energies%e_kinetic     = ekin_sum_p
    energies%e_localpsp    = epot_sum_p
    energies%e_nonlocalpsp = eproj_sum_p
    gnrm                   = gnrm_p
#endif
    residm = sqrt(gnrm / wfs%mband_init)

!    accurex = abs(eks-ekin_sum)
    write(message,*) 'ekin_sum,eks', energies%e_kinetic, wfs%ekinetic_init
    call wrtout(06, message, 'COLL')
    write(message,*) 'expected accuracy in total energy due to grid size', &
                   & abs(energies%e_kinetic - wfs%ekinetic_init)
    call wrtout(06, message, 'COLL')
    write(message,*) 'suggested value for gnrm_cv', abs(energies%e_kinetic - wfs%ekinetic_init)
    call wrtout(06, message, 'COLL')
    
    ! Solve the KS equations
    call solveKS(parallel, mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
               & wfs%mband_init, wfs%mbandp_init, wfs%keys%nvctr(0), wfs%keys%nvctr(1), &
               & wfs%mvctrp, wfs%psi_init, hpsi, wfs%psi, wfs%eval)
    deallocate(hpsi)
    deallocate(wfs%psi_init)
    deallocate(wfs%occ_init)
    wfs%mband_init = 0
    
    ! Get the density
    call sumrho(parallel, mpi_enreg%me, dtset%mband, wfs%mbandp, &
              & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
              & dtset%wvl_internal%nSize(3), dtset%wvl_hgrid, occ, &
              & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%nvctr(0), &
              & wfs%keys%nvctr(1), wfs%keys%keyg, wfs%keys%keyv, wfs%psi, rhor)
    
    return
  end if
  
  
  !Normal case (when neither first step nor wave-functions initialised from
  ! atomic orbitals.
             
  ! Apply  orthogonality constraints to all orbitals belonging to iproc
#if defined MPI
  allocate(hpsit(wfs%mvctrp, wfs%mbandp * mpi_enreg%nproc))
  call transallwaves(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                   & wfs%keys%nvctr(0), wfs%keys%nvctr(1), wfs%mvctrp, wfs%hpsi, hpsit)
  call orthoconstraint_p(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                       & occ, wfs%mvctrp, wfs%psit, hpsit, scprsum)
  call untransallwaves(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                     & wfs%keys%nvctr(0), wfs%keys%nvctr(1), wfs%mvctrp, hpsit, wfs%hpsi)
  deallocate(hpsit)
#else
  call orthoconstraint(dtset%mband, wfs%mbandp, occ, wfs%mvctrp, wfs%psi, wfs%hpsi, scprsum_p)
#endif

  ! norm of gradient
  gnrm_p = 0.d0
  do iorb = mpi_enreg%me * wfs%mbandp + 1, &
          & min((mpi_enreg%me + 1) * wfs%mbandp, dtset%mband), 1
     scpr = DNRM2(wfs%keys%nvctr(0) + 7 * wfs%keys%nvctr(1), &
          & wfs%hpsi(1, iorb - mpi_enreg%me * wfs%mbandp), 1) 
     gnrm_p = gnrm_p + scpr ** 2
  enddo

  ! Preconditions all orbitals belonging to iproc
  call preconditionall(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                     & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
                     & dtset%wvl_internal%nSize(3), dtset%wvl_internal%fineGrid(1, 1), &
                     & dtset%wvl_internal%fineGrid(2, 1), dtset%wvl_internal%fineGrid(1, 2), &
                     & dtset%wvl_internal%fineGrid(2, 2), dtset%wvl_internal%fineGrid(1, 3), &
                     & dtset%wvl_internal%fineGrid(2, 3), dtset%wvl_hgrid, &
                     & dtset%wvl_nprccg, wfs%keys%nseg(0), wfs%keys%nseg(1), &
                     & wfs%keys%nvctr(0), wfs%keys%nvctr(1), wfs%keys%keyg, &
                     & wfs%keys%keyv, wfs%eval, wfs%ibyz_c, wfs%ibxz_c, wfs%ibxy_c, &
                     & wfs%ibyz_f, wfs%ibxz_f, wfs%ibxy_f, wfs%hpsi)

  ! Reduce values in parallel case
#if defined MPI
  wrkallred(1, 2) = epot_sum_p
  wrkallred(2, 2) = ekin_sum_p
  wrkallred(3, 2) = eproj_sum_p
  wrkallred(4, 2) = gnrm_p
  wrkallred(5, 2) = scprsum_p
  call MPI_ALLREDUCE(wrkallred(1,2), wrkallred(1,1), 5, &
                   & MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, ierr)
  epot_sum  = wrkallred(1, 1)
  ekin_sum  = wrkallred(2, 1)
  eproj_sum = wrkallred(3, 1)
  gnrm      = wrkallred(4, 1)
  scprsum   = wrkallred(5, 1)
#else
  epot_sum  = epot_sum_p
  ekin_sum  = ekin_sum_p
  eproj_sum = eproj_sum_p
  gnrm      = gnrm_p
  scprsum   = scprsum_p
#endif
  residm = sqrt(gnrm / dtset%mband)

  ! WARNING! e_hartree is taken from the previous iteration as e_xc
  ! Update physical values
  energies%e_kinetic = ekin_sum
  energies%e_localpsp = epot_sum - real(2., dp) * energies%e_hartree
  energies%e_nonlocalpsp = eproj_sum
  energies%e_corepsp = real(0., dp)
  energy = energies%e_kinetic + energies%e_localpsp + energies%e_nonlocalpsp + &
         & energies%e_hartree + energies%e_xc - energies%e_vxc + &
         & energies%e_ionion + energies%e_corepsp

  ! Store the psi and its gradient for DIIS.
  if (dtset%nwfshist > 0) then
#if defined MPI
    call transallwaves(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                     & wfs%keys%nvctr(0), wfs%keys%nvctr(1), wfs%mvctrp, wfs%hpsi, &
                     & wfs%hpsidst(:,:,mids))
    do iorb = 1, dtset%mband
      do k = 1, wfs%mvctrp
        wfs%psidst(k, iorb, mids) = wfs%psit(k, iorb) 
      end do
    end do
  else
    allocate(hpsit(wfs%mvctrp, wfs%mbandp * mpi_enreg%nproc))
    call transallwaves(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                     & wfs%keys%nvctr(0), wfs%keys%nvctr(1), wfs%mvctrp, wfs%hpsi, hpsit)
#else
    do iorb = 1, dtset%mband
      do k = 1, wfs%mvctrp
        wfs%psidst(k, iorb, mids)  = wfs%psi(k, iorb)
        wfs%hpsidst(k, iorb, mids) = wfs%hpsi(k, iorb)
      enddo
    enddo
#endif
  endif

  ! We minimize (steppest descent or DIIS)
  if (dtset%nwfshist > 0) then 
#if defined MPI
    call diisstp(.true., dtset%mband, wfs%mbandp, mpi_enreg%nproc, mpi_enreg%me, &
               & wfs%ads, istepDIIS, mids, dtset%nwfshist, wfs%mvctrp, wfs%psit, &
               & wfs%psidst, wfs%hpsidst)
#else
    call diisstp(.false., dtset%mband, wfs%mbandp, mpi_enreg%nproc, mpi_enreg%me, &
               & wfs%ads, istepDIIS, mids, dtset%nwfshist, wfs%mvctrp, wfs%psi, &
               & wfs%psidst, wfs%hpsidst)
#endif
  else
    ! update mixing coefficient
    if (istep > 2) then
       if (energy > etotal_local) then
          alpha = max(.125d0, .5d0 * alpha)
          if (alpha == .125d0) then
             write(message,*) 'Convergence problem or limit'
             call wrtout(06, message, 'COLL')
             call leave_new('COLL')
          end if
       else
          alpha = min(1.05d0 * alpha, 2.d0)
       end if
    end if
    write(message,*) 'alpha = ', alpha
    call wrtout(06, message, 'COLL')
    ! update local total energy
    etotal_local = energy
    
    ! update all wavefunctions (belonging to iproc)  with the preconditioned gradient
#if defined MPI
    do iorb = 1, dtset%mband, 1
      call DAXPY(wfs%mvctrp, -alpha, hpsit(1, iorb), 1, wfs%psit(1, iorb), 1)
    enddo
    deallocate(hpsit)
#else
    do iorb = 1, dtset%mband, 1
      call DAXPY(wfs%mvctrp, -alpha, wfs%hpsi(1,iorb), 1, wfs%psi(1,iorb), 1)
    enddo
#endif
  endif


  ! Orthogonalisation
#if defined MPI
  call orthon_p(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, &
              & wfs%mbandp, wfs%mvctrp, wfs%psit)
#else
  call orthon(dtset%mband, wfs%mbandp, wfs%mvctrp, wfs%psi)
#endif

  if (abs(ekin_sum + epot_sum + eproj_sum - scprsum) > 1.d-10) then 
    write(message, '(a,a,a,a,a,E15.6,E15.6)' ) ch10,&
  &  ' wvl_vtorho :  BUG -',ch10,&
  &  '  inconsistency between gradient and energy : ', ch10, &
  &  ekin_sum + epot_sum + eproj_sum, scprsum
    call wrtout(06, message, 'COLL')
!    call leave_new('COLL')
  endif

#if defined MPI
  call untransallwaves(mpi_enreg%me, mpi_enreg%nproc, dtset%mband, wfs%mbandp, &
                     & wfs%keys%nvctr(0), wfs%keys%nvctr(1), wfs%mvctrp, wfs%psit, wfs%psi)
#endif
  
  ! Density from wavefunctions.
  call sumrho(parallel, mpi_enreg%me, dtset%mband, wfs%mbandp, &
            & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
            & dtset%wvl_internal%nSize(3), dtset%wvl_hgrid, occ, &
            & wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%nvctr(0), &
            & wfs%keys%nvctr(1), wfs%keys%keyg, wfs%keys%keyv, wfs%psi, rhor)
  
  ! Debugging messages
  write(message,'(a,3(1x,e18.11))') 'ekin_sum,epot_sum,eproj_sum',  & 
                              ekin_sum,epot_sum,eproj_sum
  call wrtout(06, message, 'COLL')
  write(message,'(a,3(1x,e18.11))') 'ehart,eexcu,vexcu', &
    & energies%e_hartree,energies%e_xc,energies%e_vxc
  call wrtout(06, message, 'COLL')
  write(message,'(a,i6,2x,e19.12,1x,e9.2)') 'iter,total energy,gnrm',istep,energy,gnrm
  call wrtout(06, message, 'COLL')

  ! Compute non-local gradient for forces, if required.
  grnl(:, :) = zero
  if (dtset%optforces == 1) then
     ! Compute forces
     write(message, '(a,a)' ) 'wvl_vtorho() : compute non-local part to gradient.'
     call wrtout(6,message,'COLL')
     ! Store xcart for each atom
     allocate(xcart(3, dtset%natom))
     call xredxcart(dtset%natom, 1, rprimd, xcart, xred)
     allocate(gxyz(3, dtset%natom))
     gxyz(:,:) = zero
     ! Add the nonlocal part of the forces to grtn (BigDFT routine)
     call nonlocal_forces(mpi_enreg%me, mpi_enreg%nproc, &
          & dtset%wvl_internal%nSize(1), dtset%wvl_internal%nSize(2), &
          & dtset%wvl_internal%nSize(3), proj%nbox_c, proj%nbox_f, &
          & dtset%ntypat, dtset%natom, dtset%mband, wfs%mbandp, proj%istart, &
          & proj%nprojel, proj%nproj, dtset%typat, psps%gth_params%psppar, &
          & psps%pspcod, occ, wfs%keys%nseg(0), wfs%keys%nseg(1), wfs%keys%nvctr(0), &
          & wfs%keys%nvctr(1), proj%keys%nseg, proj%keys%nvctr, proj%proj, &
          & wfs%keys%keyg, wfs%keys%keyv, proj%keys%keyg, proj%keys%keyv, &
          & wfs%psi, xcart, psps%gth_params%radii_cf, dtset%wvl_cpmult, &
          & dtset%wvl_fpmult, dtset%wvl_hgrid, gxyz)
     deallocate(xcart)

#if defined MPI
     !NOT IMPLEMENTED YET
     ! Add up all the force contributions
     ! call MPI_ALLREDUCE(gxyz,fxyz,3*nat,MPI_DOUBLE_PRECISION,MPI_SUM,MPI_COMM_WORLD,ierr)
#else
     !do iat=1,nat
     !   fxyz(1,iat)=gxyz(1,iat)
     !   fxyz(2,iat)=gxyz(2,iat)
     !   fxyz(3,iat)=gxyz(3,iat)
     !enddo
#endif
     ! Forces should be in reduced coordinates.
     do ia = 1, dtset%natom, 1
        do igeo = 1, 3, 1
           grnl(igeo, ia) = rprimd(1, igeo) * gxyz(1, ia) + &
                & rprimd(2, igeo) * gxyz(2, ia) + &
                & rprimd(3, igeo) * gxyz(3, ia)
        end do
     end do
     deallocate(gxyz)
  end if

#else
  write(message, '(a,a,a,a)' ) ch10,&
    &  ' wvl_vtorho : BigDFT library is not compiled.', ch10, &
    &  '   Action, used the flag --enable-bigdft when configuring.'
  call wrtout(6,message,'COLL')
  call leave_new('COLL')
#endif
end subroutine wvl_vtorho
!!***
