/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.service.displayhash;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.annotation.SystemApi;
import android.graphics.Rect;
import android.hardware.HardwareBuffer;
import android.os.Parcelable;
import android.util.Size;
import android.view.displayhash.DisplayHashResultCallback;

import com.android.internal.util.DataClass;

/**
 * Information passed from the {@link DisplayHashingService} to system server about how to get the
 * display data that will be used to generate the {@link android.view.displayhash.DisplayHash}
 *
 * @hide
 */
@SystemApi
@DataClass(genAidl = true, genToString = true, genParcelable = true, genHiddenConstructor = true)
public final class DisplayHashParams implements Parcelable {
    /**
     * The size to scale the buffer to so the hash algorithm can properly generate the hash. The
     * buffer given to the {@link DisplayHashingService#onGenerateDisplayHash(byte[],
     * HardwareBuffer, Rect, String, DisplayHashResultCallback)} will be stretched based on the
     * value set here. If {@code null}, the buffer size will not be changed.
     */
    @Nullable
    private final Size mBufferSize;

    /**
     * Whether the content will be captured in grayscale or color.
     */
    private final boolean mGrayscaleBuffer;

    /**
     * A builder for {@link DisplayHashParams}
     */
    public static final class Builder {
        @Nullable
        private Size mBufferSize;
        private boolean mGrayscaleBuffer;

        /**
         * Creates a new Builder.
         */
        public Builder() {
        }

        /**
         * The size to scale the buffer to so the hash algorithm can properly generate the hash.
         */
        @NonNull
        public Builder setBufferSize(int width, int height) {
            mBufferSize = new Size(width, height);
            return this;
        }

        /**
         * Whether the content will be captured in grayscale or color.
         */
        @NonNull
        public Builder setGrayscaleBuffer(boolean grayscaleBuffer) {
            mGrayscaleBuffer = grayscaleBuffer;
            return this;
        }

        /** Builds the instance. This builder should not be touched after calling this! */
        @NonNull
        public DisplayHashParams build() {
            return new DisplayHashParams(mBufferSize, mGrayscaleBuffer);
        }
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/service/displayhash/DisplayHashParams.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new DisplayHashParams.
     *
     * @param bufferSize
     *   The size to scale the buffer to so the hash algorithm can properly generate the hash. The
     *   buffer given to the {@link DisplayHashingService#onGenerateDisplayHash(byte[],
     *   HardwareBuffer, Rect, String, DisplayHashResultCallback)} will be stretched based on the
     *   value set here. If {@code null}, the buffer size will not be changed.
     * @param grayscaleBuffer
     *   Whether the content will be captured in grayscale or color.
     * @hide
     */
    @DataClass.Generated.Member
    public DisplayHashParams(
            @Nullable Size bufferSize,
            boolean grayscaleBuffer) {
        this.mBufferSize = bufferSize;
        this.mGrayscaleBuffer = grayscaleBuffer;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * The size to scale the buffer to so the hash algorithm can properly generate the hash. The
     * buffer given to the {@link DisplayHashingService#onGenerateDisplayHash(byte[],
     * HardwareBuffer, Rect, String, DisplayHashResultCallback)} will be stretched based on the
     * value set here. If {@code null}, the buffer size will not be changed.
     */
    @DataClass.Generated.Member
    public @Nullable Size getBufferSize() {
        return mBufferSize;
    }

    /**
     * Whether the content will be captured in grayscale or color.
     */
    @DataClass.Generated.Member
    public boolean isGrayscaleBuffer() {
        return mGrayscaleBuffer;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "DisplayHashParams { " +
                "bufferSize = " + mBufferSize + ", " +
                "grayscaleBuffer = " + mGrayscaleBuffer +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull android.os.Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        byte flg = 0;
        if (mGrayscaleBuffer) flg |= 0x2;
        if (mBufferSize != null) flg |= 0x1;
        dest.writeByte(flg);
        if (mBufferSize != null) dest.writeSize(mBufferSize);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ DisplayHashParams(@NonNull android.os.Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        byte flg = in.readByte();
        boolean grayscaleBuffer = (flg & 0x2) != 0;
        Size bufferSize = (flg & 0x1) == 0 ? null : (Size) in.readSize();

        this.mBufferSize = bufferSize;
        this.mGrayscaleBuffer = grayscaleBuffer;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<DisplayHashParams> CREATOR
            = new Parcelable.Creator<DisplayHashParams>() {
        @Override
        public DisplayHashParams[] newArray(int size) {
            return new DisplayHashParams[size];
        }

        @Override
        public DisplayHashParams createFromParcel(@NonNull android.os.Parcel in) {
            return new DisplayHashParams(in);
        }
    };

    @DataClass.Generated(
            time = 1619638159453L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/service/displayhash/DisplayHashParams.java",
            inputSignatures = "private final @android.annotation.Nullable android.util.Size mBufferSize\nprivate final  boolean mGrayscaleBuffer\nclass DisplayHashParams extends java.lang.Object implements [android.os.Parcelable]\nprivate @android.annotation.Nullable android.util.Size mBufferSize\nprivate  boolean mGrayscaleBuffer\npublic @android.annotation.NonNull android.service.displayhash.DisplayHashParams.Builder setBufferSize(int,int)\npublic @android.annotation.NonNull android.service.displayhash.DisplayHashParams.Builder setGrayscaleBuffer(boolean)\npublic @android.annotation.NonNull android.service.displayhash.DisplayHashParams build()\nclass Builder extends java.lang.Object implements []\n@com.android.internal.util.DataClass(genAidl=true, genToString=true, genParcelable=true, genHiddenConstructor=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
