/* Ergo, version 3.3, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2013 Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */

/*-*-mode: C; c-indentation-style: "bsd"; c-basic-offset: 4; -*-*/
/** @file fun-pbex.c PBEx implementation.

   Automatically generated code implementing pbex functional and its
   derivatives. Generated by func-codegen.pl being a part of a 
   "Automatic code generation framework for analytical functional
    derivative evaluation", Pawel Salek, 2004

    This functional has been generated from following input:
    ------ cut here -------
pi:3.14159265358979312;

xa:sqrt(grada*grada)/rhoa^(4/3);
xb:sqrt(gradb*gradb)/rhob^(4/3);

 parameters for pbex  
R:0.804;
d:0.066725;
mu:d*pi^2/3;
Sa:xa/(2*(6*pi^2)^(1/3)); 
Sb:xb/(2*(6*pi^2)^(1/3)); 

 functions for pbex  
F(S):=1+R-R/(1+mu*S^2/R);
Ea(n):=-3/(4*pi)*(3*pi^2)^(1/3)*n^(4/3)*F(Sa);
Eb(n):=-3/(4*pi)*(3*pi^2)^(1/3)*n^(4/3)*F(Sb);

 kernel  
K(rhoa,grada,rhob,gradb,gradab):=0.5*(Ea(2*rhoa)+Eb(2*rhob));

    ------ cut here -------
*/

 
/* strictly conform to XOPEN ANSI C standard */
#if !defined(SYS_DEC)
/* XOPEN compliance is missing on old Tru64 4.0E Alphas and pow() prototype
 * is not specified. */
#define _XOPEN_SOURCE          600
#define _XOPEN_SOURCE_EXTENDED 1
#endif
#include <math.h>
#include <stddef.h>
 
#define __CVERSION__
 
#include "functionals.h"
 
/* INTERFACE PART */
static int pbex_isgga(void) { return 1; } /* FIXME: detect! */
static int pbex_read(const char *conf_line);
static real pbex_energy(const FunDensProp* dp);
static void pbex_first(FunFirstFuncDrv *ds,   real factor,
                         const FunDensProp* dp);
static void pbex_second(FunSecondFuncDrv *ds, real factor,
                          const FunDensProp* dp);
static void pbex_third(FunThirdFuncDrv *ds,   real factor,
                         const FunDensProp* dp);
 
Functional PbexFunctional = {
  "Pbex",       /* name */
  pbex_isgga,   /* gga-corrected */
  pbex_read,
  NULL,
  pbex_energy,
  pbex_first,
  pbex_second,
  pbex_third
};
 
/* IMPLEMENTATION PART */
static int
pbex_read(const char *conf_line)
{
    fun_set_hf_weight(0);
    return 1;
}

static real
pbex_energy(const FunDensProp *dp)
{
    real res;
    real rhoa = dp->rhoa, rhob = dp->rhob;
    real grada = dp->grada, gradb = dp->gradb;


    real t1;

    t1 = POW(2.0,0.33333333333333);

   /* code */
    res = 0.5*(-1.477117532764045*t1*POW(rhob,1.333333333333333)*
        (1.804-0.804/(0.00449279948023*POW(gradb,2.0)/POW(rhob,2.666666666666667)+
        1.0))-1.477117532764045*t1*POW(rhoa,1.333333333333333)*(1.804-
        0.804/(0.00449279948023*POW(grada,2.0)/POW(rhoa,2.666666666666667)+
        1.0)));

    return res;
}

static void
pbex_first_helper(real rhoa, real grada, real *res)
{
    real t1, t2, t3, t4;

    t1 = POW(2.0,0.33333333333333);
    t2 = POW(grada,2.0);
    t3 = 0.00449279948023*t2/POW(rhoa,2.666666666666667)+
        1.0;
    t4 = 1/POW(t3,2.0);

   /* code */
    res[0] = 0.5*(0.0142284263421*t1*t2*t4/POW(rhoa,2.333333333333334)-
        1.969490043685393*t1*(1.804-0.804/t3)*POW(rhoa,0.33333333333333));
    res[1] = -
        0.00533565987829*t1*t4*grada/POW(rhoa,1.333333333333333);
}

static void
pbex_first(FunFirstFuncDrv *ds, real factor, const FunDensProp *dp)
{
    real res[2];

    pbex_first_helper(dp->rhoa, dp->grada, res);
   /* Final assignment */
    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];

    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pbex_first_helper(dp->rhob, dp->gradb, res);
    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];
}

static void
pbex_second_helper(real rhoa, real grada, real *res)
{
    real t1, t2, t3, t4, t5, t6, t7, t8;

    t1 = POW(2.0,0.33333333333333);
    t2 = POW(grada,2.0);
    t3 = 0.00449279948023*t2/POW(rhoa,2.666666666666667)+
        1.0;
    t4 = 1/POW(t3,2.0);
    t5 = 1/POW(rhoa,2.333333333333334);
    t6 = 1.804-0.804/t3;
    t7 = 1/POW(rhoa,1.333333333333333);
    t8 = 1/POW(t3,3.0);

   /* code */
    res[0] = 0.5*(0.0142284263421*t1*t2*t4*t5-1.969490043685393*
        t1*t6*POW(rhoa,0.33333333333333));
    res[1] = -0.00533565987829*t1*grada*t4*t7;
    res[2] = 0.5*(3.409358211962494E-4*t1*t8*POW(grada,4.0)/
        POW(rhoa,6.0)-0.0142284263421*t1*t2*t4/POW(rhoa,3.333333333333334)-
        0.65649668122846*t1*t6/POW(rhoa,0.66666666666667));
    res[3] = 0.5*(0.0142284263421*t1*grada*t4*t5-2.557018658971871E-4*
        t1*t8*POW(grada,3.0)/POW(rhoa,5.0));
    res[4] = 9.588819971144516E-5*t1*t2*t8/POW(rhoa,4.0)-
        0.00533565987829*t1*t4*t7;

}

static void
pbex_second(FunSecondFuncDrv *ds, real factor, const FunDensProp* dp)
{
    real res[5];
 
    pbex_second_helper(dp->rhoa, dp->grada, res);

    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];
    ds->df2000 += factor*res[2];
    ds->df1010 += factor*res[3];
    ds->df0020 += factor*res[4];

    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pbex_second_helper(dp->rhob, dp->gradb, res);
    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];
    ds->df0200 += factor*res[2];
    ds->df0101 += factor*res[3];
    ds->df0002 += factor*res[4];
}

static void
pbex_third_helper(real rhoa, real grada, real *res)
{
    real t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
    real t11, t12, t13, t14, t15;

    t1 = POW(2.0,0.33333333333333);
    t2 = POW(grada,2.0);
    t3 = 0.00449279948023*t2/POW(rhoa,2.666666666666667)+
        1.0;
    t4 = 1/POW(t3,2.0);
    t5 = 1/POW(rhoa,2.333333333333334);
    t6 = 1.804-0.804/t3;
    t7 = 1/POW(rhoa,1.333333333333333);
    t8 = POW(grada,4.0);
    t9 = 1/POW(t3,3.0);
    t10 = 1/POW(rhoa,6.0);
    t11 = 1/POW(rhoa,3.333333333333334);
    t12 = POW(grada,3.0);
    t13 = 1/POW(rhoa,5.0);
    t14 = 1/POW(rhoa,4.0);
    t15 = 1/POW(t3,4.0);

   /* code */
    res[0] = 0.5*(0.0142284263421*t1*t2*t4*t5-1.969490043685393*
        t1*t6*POW(rhoa,0.33333333333333));
    res[1] = -0.00533565987829*t1*grada*t4*t7;
    res[2] = 0.5*(-0.65649668122846*t1*t6/POW(rhoa,0.66666666666667)-
        0.0142284263421*t1*t2*t4*t11+3.409358211962494E-4*t1*t8*t9*
        t10);
    res[3] = 0.5*(0.0142284263421*t1*grada*t4*t5-2.557018658971871E-4*
        t1*t12*t9*t13);
    res[4] = 9.588819971144516E-5*t1*t2*t9*t14-0.00533565987829*
        t1*t4*t7;
    res[5] = 0.5*(-9.190537681576017E-6*t1*t15*POW(grada,
        5.0)/POW(rhoa,8.666666666666666)-0.0331996614649*t1*grada*
        t4*t11+0.00161944515068*t1*t12*t9*t10);
    res[6] = 0.5*(6.892903261182013E-6*t1*t15*t8/POW(rhoa,
        7.666666666666667)+0.0142284263421*t1*t4*t5-0.00102280746359*
        t1*t2*t9*t13);
    res[7] = 0.5*(1.225405024210135E-5*t1*t15*POW(grada,6.0)/
        POW(rhoa,9.666666666666666)-0.00238655074837*t1*t8*t9/POW(rhoa,
        7.0)+0.05375183284794*t1*t2*t4/POW(rhoa,4.333333333333333)+
        0.43766445415231*t1*t6/POW(rhoa,1.666666666666667));
    res[8] = 2.876645991343355E-4*t1*grada*t9*t14-2.584838722943255E-6*
        t1*t12*t15/POW(rhoa,6.666666666666667);

}

static void
pbex_third(FunThirdFuncDrv *ds, real factor, const FunDensProp* dp)
{
    real res[9];
 
    pbex_third_helper(dp->rhoa, dp->grada, res);

    ds->df1000 += factor*res[0];
    ds->df0010 += factor*res[1];

    ds->df2000 += factor*res[2];
    ds->df1010 += factor*res[3];
    ds->df0020 += factor*res[4];

    ds->df2010 += factor*res[5];
    ds->df1020 += factor*res[6];
    ds->df3000 += factor*res[7];
    ds->df0030 += factor*res[8];

    if(FABS(dp->rhoa-dp->rhob)>1e-13 ||
       FABS(dp->grada-dp->gradb)>1e-13)
        pbex_third_helper(dp->rhob, dp->gradb, res);

    ds->df0100 += factor*res[0];
    ds->df0001 += factor*res[1];

    ds->df0200 += factor*res[2];
    ds->df0101 += factor*res[3];
    ds->df0002 += factor*res[4];

    ds->df0201 += factor*res[5];
    ds->df0102 += factor*res[6];
    ds->df0300 += factor*res[7];
    ds->df0003 += factor*res[8];
}
