/* Copyright 2008 Simon Richter <Simon.Richter@hogyros.de>
 *
 * Released under the Boost Software Licence 1.0
 */

#ifndef util_parser_hpp_
#define util_parser_hpp_ 1

#include <boost/intrusive_ptr.hpp>

namespace util {

template<typename T>
void intrusive_ptr_add_ref(T *i, typename T::is_parser_impl * = 0)
{
	++i->refcount;
}

template<typename T>
void intrusive_ptr_release(T *i, typename T::is_parser_impl * = 0)
{
	if(!--i->refcount)
		delete i;
}

template<typename T>
class parser
{
public:
	class impl;

	parser(void) : slave(0) { }
	parser(boost::intrusive_ptr<impl> const &slave) : slave(slave) { }

	typedef T value_type;
	typedef std::forward_iterator_tag iterator_category;
	typedef void difference_type;
	typedef T const *pointer;
	typedef T const &reference;

	bool operator==(parser<T> const &rhs) const
	{
		return !slave && !rhs.slave;
	}
	bool operator!=(parser<T> const &rhs) const
	{
		return slave || rhs.slave;
	}

	pointer operator->(void) const
	{
		return slave->operator->();
	}
	reference operator*(void) const
	{
		return slave->operator*();
	}

	parser<T> &operator++(void)
	{
		slave->operator++();
		if(!*slave)
			slave = 0;
		return *this;
	}

	class impl
	{
	public:
		impl(void) throw() : refcount(0) { }
		virtual ~impl(void) throw() { }

		typedef typename parser<T>::value_type value_type;
		typedef typename parser<T>::pointer pointer;
		typedef typename parser<T>::reference reference;

		virtual pointer operator->(void) const = 0;
		virtual reference operator*(void) const = 0;
		virtual impl &operator++(void) = 0;
		virtual bool operator!(void) const = 0;

		operator void const *(void) const
		{
			if(!*this)
				return 0;
			return this;
		}

		typedef void is_parser_impl;

	private:
		unsigned int refcount;

		friend void intrusive_ptr_add_ref<>(impl *, is_parser_impl *);
		friend void intrusive_ptr_release<>(impl *, is_parser_impl *);
	};

private:
	boost::intrusive_ptr<impl> slave;
};

}

#endif
