//*******************************************************************
// Copyright (C) 2000 ImageLinks Inc. 
//
// License:  LGPL
// 
// See LICENSE.txt file in the top level directory for more details.
//
// Author: Garrett Potts
//
//*************************************************************************
// $Id: ossimBumpShadeTileSource.cpp 13382 2008-08-04 18:53:26Z gpotts $

#include <ossim/imaging/ossimBumpShadeTileSource.h>
#include <ossim/imaging/ossimImageDataFactory.h>
#include <ossim/imaging/ossimImageData.h>
#include <ossim/imaging/ossimTilePatch.h>
#include <ossim/imaging/ossimImageData.h>
#include <ossim/base/ossimNumericProperty.h>
#include <ossim/base/ossimKeywordlist.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/base/ossimKeyword.h>
#include <ossim/base/ossimMatrix3x3.h>
#include <ossim/base/ossimRgbVector.h>
#include <ossim/imaging/ossimImageToPlaneNormalFilter.h>

RTTI_DEF1(ossimBumpShadeTileSource,
          "ossimBumpShadeTileSource",
          ossimImageCombiner);


ossimBumpShadeTileSource::ossimBumpShadeTileSource()
   :ossimImageCombiner(NULL, 2, 0, true, false),
    theTile(NULL),
    theLightSourceElevationAngle(45.0),
    theLightSourceAzimuthAngle(45.0),
    theLightDirection(3)
{
   initialize();
}

ossimBumpShadeTileSource::~ossimBumpShadeTileSource()
{
}

ossimRefPtr<ossimImageData> ossimBumpShadeTileSource::getTile(
   const  ossimIrect& tileRect,
   ossim_uint32 resLevel)
{
   if(!getInput(0)) return NULL;

   if(!theTile.get())
   {
      allocate();
   }
   if(!theTile.valid())
   {
      return theTile;
   }
   ossimImageSource* normalSource =
      PTR_CAST(ossimImageSource, getInput(0));
   ossimImageSource* colorSource =
      PTR_CAST(ossimImageSource, getInput(1));
   
   if(!theTile.get())
   {
      return ossimRefPtr<ossimImageData>();
   }

   theTile->setImageRectangle(tileRect);
 
   ossimRefPtr<ossimImageData> inputTile = NULL;

   if(isSourceEnabled())
   {
      theTile->makeBlank();
      
      if(colorSource)
      {
         
         inputTile = colorSource->getTile(tileRect, resLevel);
      }
      ossimRefPtr<ossimImageData> normalData = normalSource->getTile(tileRect, resLevel);
      if(!normalData)
      {
         return ossimRefPtr<ossimImageData>();
      }
      if ( (normalData->getDataObjectStatus() == OSSIM_NULL) ||
           (normalData->getDataObjectStatus() == OSSIM_EMPTY)||
           (normalData->getNumberOfBands() != 3)||
           (normalData->getScalarType() != OSSIM_DOUBLE))
      {
         return ossimRefPtr<ossimImageData>();
      }
      ossim_float64* normalBuf[3];
      normalBuf[0] = static_cast<ossim_float64*>(normalData->getBuf(0));
      normalBuf[1] = static_cast<ossim_float64*>(normalData->getBuf(1));
      normalBuf[2] = static_cast<ossim_float64*>(normalData->getBuf(2));
      ossim_float64 normalNp = normalData->getNullPix(0);
      // if we have some color data then use it for the bump
      // else we will default to a grey scale bump shade
      //
      if(inputTile.get() &&
         (inputTile->getDataObjectStatus() != OSSIM_EMPTY) &&
         (inputTile->getDataObjectStatus() != OSSIM_NULL))
         
      {
         switch(inputTile->getScalarType())
         {
            case OSSIM_UCHAR:
            {
               ossim_uint8* resultBuf[3];
               ossim_uint8* colorBuf[3];
               resultBuf[0] = static_cast<ossim_uint8*>(theTile->getBuf(0));
               resultBuf[1] = static_cast<ossim_uint8*>(theTile->getBuf(1));
               resultBuf[2] = static_cast<ossim_uint8*>(theTile->getBuf(2));
               colorBuf[0]  = static_cast<ossim_uint8*>(inputTile->getBuf(0));
               if(inputTile->getBuf(1))
               {
                  colorBuf[1] = static_cast<ossim_uint8*>(inputTile->getBuf(1));
               }
               else
               {
                  colorBuf[1] = colorBuf[0];
               }
               if(inputTile->getBuf(2))
               {
                  colorBuf[2] = static_cast<ossim_uint8*>(inputTile->getBuf(2));
               }
               else
               {
                  colorBuf[2] = colorBuf[0];
               }
            
               long h = theTile->getHeight();
               long w = theTile->getWidth();
               for(long y = 0; y < h; ++y)
               {
                  for(long x = 0; x < w; ++x)
                  {
                     if((*normalBuf[0] != normalNp) &&
                        (*normalBuf[1] != normalNp) &&
                        (*normalBuf[2] != normalNp) )
                     {
                        if((*colorBuf[0])||(*colorBuf[1])||(*colorBuf[2]))
                        {
                           computeColor(*resultBuf[0],
                                        *resultBuf[1],
                                        *resultBuf[2],
                                        *normalBuf[0],
                                        *normalBuf[1],
                                        *normalBuf[2],
                                        *colorBuf[0],
                                        *colorBuf[1],
                                        *colorBuf[2]);
                        }
                        else 
                        {
                           computeColor(*resultBuf[0],
                                        *resultBuf[1],
                                        *resultBuf[2],
                                        *normalBuf[0],
                                        *normalBuf[1],
                                        *normalBuf[2],
                                        255,
                                        255,
                                        255);
                        }
                     }
                     else
                     {
                        *resultBuf[0] = *colorBuf[0];
                        *resultBuf[1] = *colorBuf[1];
                        *resultBuf[2] = *colorBuf[2];
                     }
                     resultBuf[0]++;
                     resultBuf[1]++;
                     resultBuf[2]++;
                     colorBuf[0]++;
                     colorBuf[1]++;
                     colorBuf[2]++;
                     normalBuf[0]++;
                     normalBuf[1]++;
                     normalBuf[2]++;
                  }
               }
               break;
            }
            default:
            {
               ossimNotify(ossimNotifyLevel_NOTICE)
                  << "ossimBumpShadeTileSource::getTile NOTICE:\n"
                  << "only 8-bit unsigned char is supported." << endl;
            }
         }
      }
      else
      {
         ossim_uint8* resultBuf[3];
         resultBuf[0] = static_cast<ossim_uint8*>(theTile->getBuf(0));
         resultBuf[1] = static_cast<ossim_uint8*>(theTile->getBuf(1));
         resultBuf[2] = static_cast<ossim_uint8*>(theTile->getBuf(2));
         long h = theTile->getHeight();
         long w = theTile->getWidth();
         for(long y = 0; y < h; ++y)
         {
            for(long x = 0; x < w; ++x)
            {
               if((*normalBuf[0] != normalNp) &&
                  (*normalBuf[1] != normalNp) &&
                  (*normalBuf[2] != normalNp) )
               {
                     computeColor(*resultBuf[0],
                                  *resultBuf[1],
                                  *resultBuf[2],
                                  *normalBuf[0],
                                  *normalBuf[1],
                                  *normalBuf[2],
                                  (ossim_uint8)255,
                                  (ossim_uint8)255,
                                  (ossim_uint8)255);
               }
               else
               {
                  *resultBuf[0] = 0;
                  *resultBuf[1] = 0;
                  *resultBuf[2] = 0;
               }
               resultBuf[0]++;
               resultBuf[1]++;
               resultBuf[2]++;
               normalBuf[0]++;
               normalBuf[1]++;
               normalBuf[2]++;
            }
         }
      }      
   }
   theTile->validate();
   return theTile;
}

void ossimBumpShadeTileSource::computeColor(ossim_uint8& r,
                                            ossim_uint8& g,
                                            ossim_uint8& b,
                                            ossim_float64 normalX,
                                            ossim_float64 normalY,
                                            ossim_float64 normalZ,
                                            ossim_uint8 dr,
                                            ossim_uint8 dg,
                                            ossim_uint8 db)const
{
   double c = normalX*theLightDirection[0] +
              normalY*theLightDirection[1] +
              normalZ*theLightDirection[2];
   
   if(fabs(c) > FLT_EPSILON)
   {
      r = ossimRgbVector::clamp(ossim::round<int>(c*dr), 1, 255);
      g = ossimRgbVector::clamp(ossim::round<int>(c*dg), 1, 255);
      b = ossimRgbVector::clamp(ossim::round<int>(c*db), 1, 255);
   }
   else
   {
      r = 1;
      g = 1;
      b = 1;
   }
}

ossimIrect ossimBumpShadeTileSource::getBoundingRect(ossim_uint32 resLevel)const
{
   ossimImageSource* source = PTR_CAST(ossimImageSource, getInput(0));
   if(source)
   {
      return source->getBoundingRect(resLevel);
   }
   ossimDrect rect;
   rect.makeNan();
   return rect;
}


void ossimBumpShadeTileSource::getDecimationFactor(ossim_uint32 resLevel,
                                                   ossimDpt& result) const
{
   ossimImageSource* source = PTR_CAST(ossimImageSource, getInput(0));
   if(source)
   {
      return source->getDecimationFactor(resLevel, result);
   }
   
   result.makeNan();
}

void ossimBumpShadeTileSource::getDecimationFactors(vector<ossimDpt>& decimations) const
{
   ossimImageSource* source = PTR_CAST(ossimImageSource, getInput(0));
   
   if(source)
   {
      return source->getDecimationFactors(decimations);
   }
}

ossim_uint32 ossimBumpShadeTileSource::getNumberOfDecimationLevels()const
{
   ossimImageSource* source = PTR_CAST(ossimImageSource, getInput(0));
   
   if(source)
   {
      return source->getNumberOfDecimationLevels();
   }
   
   return 0;
}

ossim_uint32 ossimBumpShadeTileSource::getTileWidth()const
{
   ossimImageSource* source = PTR_CAST(ossimImageSource, getInput(0));
   if(source)
   {
      return source->getTileWidth();
   }
   
   return 128;
}

ossim_uint32 ossimBumpShadeTileSource::getTileHeight()const
{
   ossimImageSource* source = PTR_CAST(ossimImageSource, getInput(0));
   if(source)
   {
      return source->getTileHeight();
   }
   
   return 128;
}

void ossimBumpShadeTileSource::initialize()
{
  ossimImageCombiner::initialize();
   theTile = 0;
   
  computeLightDirection();
  
}

void ossimBumpShadeTileSource::allocate()
{
   theTile = ossimImageDataFactory::instance()->create(this, this);
   theTile->initialize();
}

void ossimBumpShadeTileSource::computeLightDirection()
{
   NEWMAT::Matrix m = ossimMatrix3x3::createRotationMatrix(theLightSourceElevationAngle,
                                                           0.0,
                                                           theLightSourceAzimuthAngle);
   NEWMAT::ColumnVector v(3);
   v[0] = 0;
   v[1] = 1;
   v[2] = 0;
   v = m*v;
   // reflect Z.  We need the Z pointing up from the surface and not into it.
   //
   ossimColumnVector3d d(v[0], v[1], -v[2]);
   d = d.unit();
   theLightDirection[0] = d[0];
   theLightDirection[1] = d[1];
   theLightDirection[2] = d[2];
}

bool ossimBumpShadeTileSource::loadState(const ossimKeywordlist& kwl,
                                         const char* prefix)
{
   const char* elevationAngle = kwl.find(prefix, ossimKeywordNames::ELEVATION_ANGLE_KW);
   const char* azimuthAngle   = kwl.find(prefix, ossimKeywordNames::AZIMUTH_ANGLE_KW);

   
   if(elevationAngle)
   {
      theLightSourceElevationAngle = ossimString(elevationAngle).toDouble();
   }

   if(azimuthAngle)
   {
      theLightSourceAzimuthAngle = ossimString(azimuthAngle).toDouble();
   }

   computeLightDirection();

   bool result = ossimImageSource::loadState(kwl, prefix);

   theInputListIsFixedFlag  = true;
   theOutputListIsFixedFlag = false;
   if(!getNumberOfInputs()) setNumberOfInputs(2);
   
   return result;
}

bool ossimBumpShadeTileSource::saveState(ossimKeywordlist& kwl,
                                         const char* prefix)const
{
   kwl.add(prefix,
           ossimKeywordNames::ELEVATION_ANGLE_KW,
           theLightSourceElevationAngle,
           true);

   kwl.add(prefix,
           ossimKeywordNames::AZIMUTH_ANGLE_KW,
           theLightSourceAzimuthAngle,
           true);
   
   return ossimImageSource::saveState(kwl, prefix);
}

ossimString ossimBumpShadeTileSource::getShortName()const
{
   return ossimString("bump shader");
}

ossimString ossimBumpShadeTileSource::getLongName()const
{
   return ossimString("Blinn's bump map filter");
}

ossim_uint32 ossimBumpShadeTileSource::getNumberOfOutputBands() const
{
   return 3;
}

ossimScalarType ossimBumpShadeTileSource::getOutputScalarType() const
{
   return OSSIM_UCHAR;
}

double ossimBumpShadeTileSource::getNullPixelValue()const
{
   return 0.0;
}

double ossimBumpShadeTileSource::getMinPixelValue(ossim_uint32 /* band */)const
{
   return 1.0;
}

double ossimBumpShadeTileSource::getMaxPixelValue(ossim_uint32 /* band */)const
{
   return 255.0;
}

double ossimBumpShadeTileSource::getAzimuthAngle()const
{
   return theLightSourceAzimuthAngle;
}

double ossimBumpShadeTileSource::getElevationAngle()const
{
   return theLightSourceElevationAngle;
}

void ossimBumpShadeTileSource::setAzimuthAngle(double angle)
{
   theLightSourceAzimuthAngle = angle;
}

void ossimBumpShadeTileSource::setElevationAngle(double angle)
{
   theLightSourceElevationAngle = angle;
}

bool ossimBumpShadeTileSource::canConnectMyInputTo(ossim_int32 inputIndex,
                                                   const ossimConnectableObject* object)const
{
   return (object&&
           ( (inputIndex>=0) && inputIndex < 2)&&
           PTR_CAST(ossimImageSource, object));
   
}

void ossimBumpShadeTileSource::connectInputEvent(ossimConnectionEvent& event)
{
   initialize();
}

void ossimBumpShadeTileSource::disconnectInputEvent(ossimConnectionEvent& event)
{
   initialize();
}

void ossimBumpShadeTileSource::propertyEvent(ossimPropertyEvent& event)
{
   initialize();
}

void ossimBumpShadeTileSource::refreshEvent(ossimRefreshEvent& event)
{
   initialize();
}

void ossimBumpShadeTileSource::setProperty(ossimRefPtr<ossimProperty> property)
{
   ossimString name = property->getName();
   if(name == "lightSourceElevationAngle")
   {
      theLightSourceElevationAngle = property->valueToString().toDouble();
   }
   else if(name == "lightSourceAzimuthAngle")
   {
      theLightSourceAzimuthAngle = property->valueToString().toDouble();
   }
   else
   {
      ossimImageCombiner::setProperty(property);
   }
}

ossimRefPtr<ossimProperty> ossimBumpShadeTileSource::getProperty(const ossimString& name)const
{
   if(name == "lightSourceElevationAngle")
   {
      ossimProperty* prop = new ossimNumericProperty(name, theLightSourceElevationAngle, 0.0, 90.0);
      prop->setCacheRefreshBit();
      return prop;
   }
   else if(name == "lightSourceAzimuthAngle")
   {
      ossimProperty* prop = new ossimNumericProperty(name, theLightSourceAzimuthAngle, 0.0, 90.0);
      prop->setCacheRefreshBit();
      return prop;
   }
   
   return ossimImageCombiner::getProperty(name);
}

void ossimBumpShadeTileSource::getPropertyNames(std::vector<ossimString>& propertyNames)const
{
   ossimImageCombiner::getPropertyNames(propertyNames);
   propertyNames.push_back("lightSourceElevationAngle");
   propertyNames.push_back("lightSourceAzimuthAngle");
}

